/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  directories.c                                                            */
/*                                                                           */
/*                                                                           */
/*  Functions to retrieve and verify CTOS disk volume directory structures.  */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  05/16/91  121J.05  C. Naik       /  Bug fix for handling insufficient    */
/*                                      memory in the verfication phase.     */
/*  05/07/91  121J.04  C. Naik       /  Bug fix for EXTRA large! directories */
/*  03/21/91  121H.03  P. Johansson  /  Save MFD information in archive      */
/*                                      dataset so that empty directories    */
/*                                      may be recreated (Volume Archive).   */
/*  01/30/91  121F.02  P. Johansson  /  Bug fix for VERY large directories.  */
/*  01/09/91  121F.01  P. Johansson  /  For real-mode CTOS, can't use 0xFFFF */
/*                                      with DeallocMemorySL.                */
/*  12/17/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define AllocMemorySL
#define BuildFileSpec
#define CloseFile
#define DeallocMemorySL
#define NlsULCmpB
#define ParseFileSpec
#define OpenFile
#define Read

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define FhbType
#define sdType
#define vhb_type

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "archive.h"
#include "archive_msgs.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

extern void create_archive_sentinel(unsigned long file_length);
extern void hex_dump(void *buffer, unsigned length, unsigned blackouts,
                     unsigned blackout_floor[], unsigned blackout_ceiling[]);
extern log_msg(unsigned nls_msg_index, unsigned signature, sdType nls_parms[],
               unsigned nls_parms_len);
extern void quicksort(char **key, unsigned n);
extern unsigned read_fhb(unsigned fhb_num);
extern unsigned read_pages(void *buffer, unsigned long vda,
                           unsigned page_offset, unsigned pages);
extern void release_disk_run(unsigned long vda, unsigned long length);

/* External variables imported by this module */

extern FhbType fhb;
extern filesys_type filesys;
extern path_type path;
extern vhb_type vhb;
extern vlpb_type vlpb;

/* Static variables global within this manuscript */

private char (*directory[12])[PAGE_SIZE];
private unsigned directory_alloc_length[12];
private unsigned long directory_index_length;
private char **filename;
private mfd_entry_type **mfd_entry = NULL;
private char (*mfd)[PAGE_SIZE] = NULL;
private unsigned mfd_pages;
private unsigned n_mfd_entry;

/* Function prototypes defined before the functions themselves are declared */

void get_directory_memory(unsigned directory_pages);
void release_directory_memory(void);
void release_mfd();

pragma Page(1);
/*-----------------------------------------------------------------------------
 This procedure is used to examine the directories poited to by MFD.Sys to
 determine the amount of memory that will subsequently be needed to load the
 MFD and the directories into memory.  If the directories themselves are also
 to be verified (Volume Archive, only) don't forget to add memory for reading
 the individual directories into memory. */

#define MAX_FILENAMES_PER_DIRECTORY_PAGE ((PAGE_SIZE - 1) / 4)

unsigned long scan_mfd(unsigned long mfd_vda, unsigned mfd_pages,
                       Boolean verify_directories) {

   unsigned i, max_directory_pages = 0, n_mfd_entry = 0, page;
   char mfd_page[PAGE_SIZE];
   unsigned long memory_needed = mfd_pages * PAGE_SIZE;

   for (page = 0; page < mfd_pages; page++)
      if (read_pages(mfd_page, mfd_vda, page, 1) == ercOK) {
         i = 1;				/* Start just past overflow flag */
         while (i < sizeof(mfd_page) && mfd_page[i] != 0) {
            memory_needed +=  sizeof(mfd_entry_type);
            max_directory_pages = _max(max_directory_pages,
                                    ((mfd_entry_type *) &mfd_page[i])->cPages);
            i += sizeof(mfd_entry_type);
         }
      }
   if (verify_directories)
      memory_needed += max_directory_pages * MAX_FILENAMES_PER_DIRECTORY_PAGE
                         * sizeof(filename)
                        + max_directory_pages * PAGE_SIZE;
   return(memory_needed);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 MFD.Sys needs to be fetched from disk volumes by both Volume Archive and
 Selective Archive.  In the first case, I/O errors encountered while reading
 the data are to be reported.  In the second, the I/O errors are not fatal---
 they just mean that useful directory information (password, default
 protection level and size) will not be available to be written to the archive
 medium when a file is stored.  Note also that the I/O error methods are
 different depending on context.  The difference is signalled by the absence
 or presence, respectively,  of a filename string (typically "<Sys>MFD.Sys")
 in the call to this function. */

unsigned fetch_mfd(sdType *sd_filename) {

   unsigned directory_len, erc, fh, filename_len, i, mfd_spec_len, page,
      xfer_count;
   char directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH],
      mfd_spec[MAX_FILE_SPECIFICATION_LENGTH];
   sdType nls_parms[] = {(void *) &page, sizeof(page)};

   release_mfd();				/* Free memory of prior use */
   if ((erc = AllocMemorySL(vhb.cPagesMfd * PAGE_SIZE, &mfd)) != ercOK)
      return(erc);
   mfd_pages = vhb.cPagesMfd;
   if (sd_filename == NULL)
      fh = 0;
   else {
      ParseFileSpec(0, sd_filename->pb, sd_filename->cb, FALSE, NULL, NULL,
                    NULL, NULL, directory, &directory_len, filename,
                    &filename_len, NULL, NULL, FALSE, 0);
      memset(mfd_spec, 0, sizeof(mfd_spec));
      BuildFileSpec(0, mfd_spec, &mfd_spec_len, NULL, sizeof(mfd_spec), FALSE,
                    &path.node[1], path.node[0], &path.volume[1],
                    path.volume[0], directory, directory_len, filename,
                    filename_len, FALSE, NULL, 0, FALSE, 0);
      if ((erc = OpenFile(&fh, mfd_spec, mfd_spec_len, NULL, 0,
                          modeRead)) != ercOK) {
         release_mfd();
         return(erc);
      }
   }
   for (page = 0; page < mfd_pages; page++) {
      erc = (fh == 0) ? read_pages(mfd[page], vhb.lfaMfdBase, page, 1)
                      : Read(fh, mfd[page], sizeof(mfd[page]),
                             page * PAGE_SIZE, &xfer_count);
      if (erc == ercOK) {
         i = 1;				/* Start just past overflow flag */
         while (i < sizeof(mfd[page]) && mfd[page][i] != 0) {
            n_mfd_entry++;
            i += sizeof(mfd_entry_type);
         }
      } else {
         if (fh == 0)
            log_msg(NLS_MFD_PAGE_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
         memset(mfd[page], 0, sizeof(mfd[page]));
      }
   }
   CloseFile(fh);
   if (     n_mfd_entry * (unsigned long) sizeof(mfd_entry) >= SEGMENT_SIZE
         || (erc = AllocMemorySL(n_mfd_entry * sizeof(mfd_entry),
                                 &mfd_entry)) != ercOK) {
      release_mfd();
      return(erc);
   }
   n_mfd_entry = 0;
   for (page = 0; page < mfd_pages; page++) {
      i = 1;				/* Start just past overflow flag */
      while (i < sizeof(mfd[page]) && mfd[page][i] != 0) {
         mfd_entry[++n_mfd_entry - 1] = (mfd_entry_type *) &mfd[page][i];
         i += sizeof(mfd_entry_type);
      }
   }
   quicksort((char **) mfd_entry, n_mfd_entry);
   if (fh == 0 && !vlpb.suppress_backup) {	/* Save ALL MFD information */
      memset(&fhb, 0, sizeof(fhb));
      for (i = 0; i < n_mfd_entry; i++) {
         memcpy(path.directory, mfd_entry[i]->sbDirectory,
                mfd_entry[i]->sbDirectory[0] + 1);
         memcpy(path.directory_password, mfd_entry[i]->sbPassword,
                mfd_entry[i]->sbPassword[0] + 1);
         path.directory_protection = mfd_entry[i]->defaultAccessCode;
         path.directory_pages = mfd_entry[i]->cPages;
         create_archive_sentinel(0);
      }
      memset(path.directory, 0, sizeof(path.directory));
      memset(path.directory_password, 0, sizeof(path.directory_password));
      path.directory_protection = 0;
      path.directory_pages = 0;
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 A previously obtained MFD is no longer needed.  Return the memory in the same
 order that it was obtained. */

private void release_mfd() {

   if (mfd_entry != NULL) {
      DeallocMemorySL(mfd_entry, n_mfd_entry * sizeof(mfd_entry));
      mfd_entry = NULL;
   }
   if (mfd != NULL) {
      DeallocMemorySL(mfd, mfd_pages * PAGE_SIZE);
      mfd = NULL;
   }
   mfd_pages = 0;
   n_mfd_entry = 0;

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 An MFD (previously read into memory) is to be verified to see if any of its
 pages overflow and also to release the disk runs used by the directories
 pointed to by the MFD (these directories are "hidden" files and the bit map
 reconciliation would not work without this step). */

void verify_mfd(void) {

   char buffer[PAGE_SIZE];
   mfd_entry_type *mfd_entry;
   char *mfd_page;
   unsigned i, page;
   sdType nls_parms[] = {(void *) &page, sizeof(page),
                                  NULL, 0};

   log_msg(NLS_VERIFYING_MFD, 1, NULL, 0);
   for (page = 0; page < vhb.cPagesMfd; page++) {
      if (mfd == NULL) {
         if (read_pages(mfd_page = buffer, vhb.lfaMfdBase, page, 1) != ercOK) {
            log_msg(NLS_MFD_PAGE_IO_ERROR, 1, nls_parms, sizeof(nls_parms));
            memset(mfd_page, 0, PAGE_SIZE);
         }
      } else
         mfd_page = mfd[page];
      if (mfd_page[0] != 0)
         log_msg(NLS_MFD_OVERFLOW, 1, nls_parms, sizeof(nls_parms));
      i = 1;				/* Start just past overflow flag */
      while (i < PAGE_SIZE && mfd_page[i] != 0) {
         mfd_entry = (mfd_entry_type *) &mfd_page[i];
         if (mfd_entry->cPages == 0) {
            nls_parms[1].pb = (void *) &mfd_entry->sbDirectory[1];
            nls_parms[1].cb = mfd_entry->sbDirectory[0];
            filesys.invalid = TRUE;
            log_msg(NLS_MFD_ENTRY_INVALID, 1, nls_parms, sizeof(nls_parms));
         } else
            release_disk_run(mfd_entry->lfaBase,
                             mfd_entry->cPages * (unsigned long) PAGE_SIZE);
         i += sizeof(mfd_entry_type);
      }
      if (i >= PAGE_SIZE) {
         filesys.invalid = TRUE;
         log_msg(NLS_MFD_PAGE_INVALID, 1, nls_parms, sizeof(nls_parms));
      }
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Use a sorted list of directories obtained from the MFD to successively read
 each directory into memory and verify its contents.  Earlier, when the MFD
 was first read, care was taken to determine the largest directory referenced
 by the MFD in order that memory for the directory pages need be allocated
 only once.  Read each directory page into memory, check for page overflow and
 self-consistent page structure and build a list of all the filenames in the
 directory.  After all pages have been read, sort the list of filenames and
 then verify that each filename points to a valid initial File Header Block
 which in turn points back to the directory from which the filename was first
 referenced. */

void verify_directories(void) {

   char buffer[PAGE_SIZE];
   char *directory_page;
   unsigned fhb_num, files, i, j, k, page;

   log_msg(NLS_VERIFYING_DIRECTORIES, 1, NULL, 0);
   for (i = 0; i < n_mfd_entry; i++) {
      sdType nls_parms[] = {(void *) &page, sizeof(page),
                            (void *) &mfd_entry[i]->sbDirectory[1],
                                     mfd_entry[i]->sbDirectory[0],
                            (void *) &mfd_entry[i]->cPages,
                                     sizeof(mfd_entry[i]->cPages),
                            (void *) &mfd_entry[i]->defaultAccessCode,
                                     sizeof(mfd_entry[i]->defaultAccessCode),
                            (void *) &fhb_num, sizeof(fhb_num)};
                            
      log_msg(NLS_DIRECTORY, 1, nls_parms, sizeof(nls_parms));
      get_directory_memory(mfd_entry[i]->cPages);
      directory_index_length = 0;
      for (page = 0; page < mfd_entry[i]->cPages; page++) {
         directory_page = (directory[page >> 7] == NULL)
                                  ? buffer : directory[page >> 7][page & 0x7F];
         if (read_pages(directory_page, mfd_entry[i]->lfaBase,
                        page, 1) == ercOK) {
            if (directory_page[0])
               log_msg(NLS_DIRECTORY_OVERFLOW, 1, nls_parms,
                       sizeof(nls_parms));
            j = 1;			/* Start just past overflow flag */
            while (j < PAGE_SIZE && directory_page[j] != 0) {
               directory_index_length += sizeof(filename);
               j += directory_page[j] + 1 + sizeof(fhb_num);
            }
            if (j >= PAGE_SIZE) {
               filesys.invalid = TRUE;
               log_msg(NLS_DIRECTORY_PAGE_INVALID, 1, nls_parms,
                       sizeof(nls_parms));
            }
         } else
            log_msg(NLS_DIRECTORY_PAGE_IO_ERROR, 1, nls_parms,
                    sizeof(nls_parms));
      }
      if (directory[0] == NULL) {
         log_msg(NLS_CANT_SORT_FILENAMES, 1, NULL, 0);
         continue;
      }
      filename = NULL;
      if (     directory_index_length >= SEGMENT_SIZE
            || AllocMemorySL(directory_index_length, &filename) != ercOK) {
         log_msg(NLS_CANT_SORT_FILENAMES, 1, NULL, 0);
         release_directory_memory();
         continue;
      }
      files = 0;
      for (page = 0; page < mfd_entry[i]->cPages; page++) {
         directory_page = directory[page >> 7][page & 0x7F];
         j = 1;				/* Start just past overflow flag */
         while (j < PAGE_SIZE && directory_page[j] != 0) {
            filename[files++] = &directory_page[j];
            j += directory_page[j] + 1 + sizeof(fhb_num);
         }
      }
      quicksort(filename, files);
      for (j = 0; j < files; j++) {
         nls_parms[1].pb = &filename[j][1];
         nls_parms[1].cb = filename[j][0];
         log_msg(NLS_FILE_WITHOUT_DIRECTORY, 1, nls_parms, sizeof(nls_parms));
         fhb_num = *((unsigned *) &filename[j][filename[j][0] + 1]);
         if (     fhb_num == 0 || fhb_num > vhb.cPagesFileHeaders
               || (   vhb.altFileHeaderPageOffset != 0
                   && ((fhb_num / vhb.altFileHeaderPageOffset) & 1) != 0)) {
            filesys.invalid = TRUE;
            log_msg(NLS_DIRECTORY_FHB_INVALID, 1, nls_parms,
                    sizeof(nls_parms));
         } else if (read_fhb(fhb_num) == ercOK)
            if (     fhb_num != fhb.fileHeaderNum
                  || fhb.headerSequenceNum != 0
                  || filename[j][0] != fhb.fileName[0]
                  || NlsULCmpB(NULL, &filename[j][1], &fhb.fileName[1],
                               fhb.fileName[0], &k) != ercOK
                  || k != 0xFFFF) {
               filesys.invalid = TRUE;
               log_msg(NLS_DIRECTORY_FHB_INVALID, 1, nls_parms,
                       sizeof(nls_parms));
            }
      }
      release_directory_memory();
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Dynamically obtain and release working memory for each directory referenced
 by the MFD.  The first block(s) obtained contain the directory pages as they
 are read in from the disk while the second holds an array of pointers to the
 filename(s) in the directory.  This list is subsequently sorted. */

private void get_directory_memory(unsigned directory_pages) {

   unsigned long directory_length = directory_pages * PAGE_SIZE;
   unsigned i;

   memset(directory, 0, sizeof(directory));
   for (i = 0; i <=  directory_length / SEGMENT_SIZE; i++) {
      if (directory_length - (i * SEGMENT_SIZE) >= SEGMENT_SIZE)
         directory_alloc_length[i] = 0xFFFF;	/* Actually forces 64 Kb */
      else
         directory_alloc_length[i] = directory_length - (i * SEGMENT_SIZE);
      if (AllocMemorySL(directory_alloc_length[i], &directory[i]) != ercOK) {
         release_directory_memory();
         break;
      }
   }

}

private void release_directory_memory(void) {

   signed i;

   if (filename != NULL)
      DeallocMemorySL(filename, directory_index_length);
   for (i = last(directory); i >= 0; i--)	/* Return bit map memory */
      if (directory[i] != NULL)
         DeallocMemorySL(directory[i], directory_alloc_length[i]);
   memset(directory, 0, sizeof(directory));	/* Destroy the pointer */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Given a File Header block in the external 'fhb' variable, read the directory
 page it references.  Then search the page to determine if there is a
 corresponding entry that points back to the File Header Block.  If not, too
 bad!  You should probably reinitialize the volume... */

void match_directory_entry_to_fhb(void) {

   char directory_page[PAGE_SIZE];
   unsigned fhb_num, i, j;
   sdType nls_parms[] = {(void *) &fhb.dirPageLfa, sizeof(fhb.dirPageLfa),
                         (void *) &fhb.fileHeaderNum,
                                  sizeof(fhb.fileHeaderNum),
                         (void *) &fhb.dirName[1], fhb.dirName[0],
                         (void *) &fhb.fileName[1], fhb.fileName[0]};

   if (read_pages(directory_page, fhb.dirPageLfa, 0, 1) != ercOK)
      log_msg(NLS_RAW_DIRECTORY_PAGE_IO_ERROR, 1, nls_parms,
              sizeof(nls_parms));
   else {
      i = 1;				/* Skip the page overflow flag */
      while (i < sizeof(directory_page) && directory_page[i] != 0) {
         fhb_num = *((unsigned *) &directory_page[i + directory_page[i] + 1]);
         if (     fhb_num == fhb.fileHeaderNum
               && directory_page[i] == fhb.fileName[0]
               && NlsULCmpB(NULL, &directory_page[i + 1], &fhb.fileName[1],
                            fhb.fileName[0], &j) == ercOK
               && j == 0xFFFF)
            return;
         i += directory_page[i] + 1 + sizeof(fhb_num);
      }
      if (i >= sizeof(directory_page)) {
         log_msg(NLS_RAW_DIRECTORY_PAGE_INVALID, 1, nls_parms,
                 sizeof(nls_parms));
         hex_dump(directory_page, sizeof(directory_page), 0, NULL, NULL);
      }
   }
   log_msg(NLS_NO_DIRECTORY_ENTRY_FOR_FHB, 1, nls_parms, sizeof(nls_parms));
   filesys.invalid = TRUE;		/* Already exited if this isn't true */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Given a directory name, see if it is described in the MFD.Sys currently in
 memory (it is the caller's responsibility to use the fetch_mfd() function at
 appropriate times to ensure that the MFD data in memory corresponds to the
 volume containing the directory).  Note that the directory search is made
 shorter by using the already sorted list of directory names (these names are
 in fact the start of a directory entry).  If the directory entry can be found,
 return a pointer to the entry.  Otherwise, return a null pointer. */

mfd_entry_type* find_mfd_entry(char directory[]) {

   unsigned i, j;

   for (i = 0; i < n_mfd_entry; i++)
      if (     fhb.dirName[0] == mfd_entry[i]->sbDirectory[0]
            && NlsULCmpB(NULL, &directory[1], &mfd_entry[i]->sbDirectory[1],
                         directory[0], &j) == ercOK
            && j == 0xFFFF)
         return(mfd_entry[i]);
   return(NULL);

}
