/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  seq_service_scsi.c                                                       */
/*                                                                           */
/*  SCSI device procedures for CTOS Sequential Access service.               */
/*                                                                           */
/*  HISTORY: (Most recent first)                                             */
/*  ----------------------------                                             */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  06/19/91  121K.10  P. Johansson  /  Initialize CDB length before SCSI    */
/*                                      command is issued.                   */
/*  05/20/91  121J.09  P. Johansson  /  Correct errors in variable length    */
/*                                      record support.                      */
/*  05/17/91  121J.08  P. Johansson  /  Don't allow buffered mode to change  */
/*                                      in SeqAccessModeSet.                 */
/*  05/08/91  121J.07  P. Johansson  /  After a deferred error, consider     */
/*                                      sense data "fresh" until a new       */
/*                                      command is issued or an explicit     */
/*                                      SeqAccessStatus operation.           */
/*  04/30/91  121J.06  P. Johansson  /  Support 2145 SCSI half-inch tape     */
/*                                      (firmware 257-003A or newer); use    */
/*                                      MODE_EXCLUSIVE for OpenScsiPath;     */
/*                                      because mode header data and block   */
/*                                      descriptor data is critical, do not  */
/*                                      tolerate any MODE SELECT errors.     */
/*  03/28/91  121H.05  P. Johansson  /  Clean up MODE SENSE and MODE SELECT  */
/*                                      logic (mostly for DDS).              */
/*  03/07/91  121G.04  P. Johansson  /  DATA IN phase for INQUIRY command.   */
/*  01/31/91  121F.03  P. Johansson  /  Obtain write protect from MODE SENSE.*/
/*  01/10/91  121F.02  P. Johansson  /  Increase time limits for space       */
/*                                      records and space filemarks.         */
/*  01/02/91  121F.01  P. Johansson  /  New procedure to return default      */
/*                                      device buffer configuration details. */
/*  12/17/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#define debug

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define CloseFile
#define CloseScsiPath
#define OpenFile
#define OpenScsiPath
#define Request
#define Respond
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

pragma Off(List);
#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include <scsi.h>
#include "seq_service.h"
pragma Pop(List);

/* Other external functions in this application invoked by this module */

/* Error return codes used by this module */

#define ScsiErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* Request codes used by this module */

pragma Off(List);
#include <rq.h>
pragma Pop(List);

/* External variables imported by this module */

extern unsigned default_resp_exch;
extern unsigned internal_os_version;
extern unsigned serv_exch;

/* Static variables global within this manuscript */

const char request_sense_cdb[6] = {REQUEST_SENSE, 0, 0, 0,
                                   sizeof(ext_sense_type), 0};
                                   
/* Function prototypes defined before the functions themselves are declared */

unsigned long calculate_device_buffer_size(dcb_scsi_type *dcb);
unsigned issue_scsi_command(dcb_scsi_type *dcb, Boolean wait_for_response);
unsigned map_scsi_status(dcb_scsi_type *dcb, unsigned erc);
unsigned mode_select(dcb_scsi_type *dcb, mode_header6_type *mode_header,
                     block_descriptor_type *block_descriptor, void *page,
                     char page_len);
unsigned mode_sense(dcb_scsi_type *dcb, char page_code, char page_control,
                    mode_header6_type *mode_header,
                    block_descriptor_type *block_descriptor,
                    void *page, char page_len);
void read_block_limits(dcb_scsi_type *dcb, unsigned *min_block_length,
                       unsigned long *max_block_length);
unsigned request_sense(dcb_scsi_type *dcb);
unsigned scsi_io(dcb_scsi_type *dcb);
unsigned scsi_mode_query(dcb_scsi_type *dcb,
                         seq_parameters_type *seq_parameters);
unsigned scsi_status(dcb_scsi_type *dcb);
void to_intel_byte_order(char *source, char *destination,
                         unsigned destination_len);
void to_scsi_byte_order(char *source, char *destination,
                        unsigned destination_len);

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function provides device specific information about buffer capacity so
 that SeqAccessOpen may allocate buffers intelligently.  The device is queried
 to determine the size of its buffers and whether or not RECOVER BUFFERED DATA
 can be used to retrieve data from them.  Additionally, recommendations are
 made for the size and nature of the buffer pool in the Sequential Access
 service itself (in order to optimize performance). */

unsigned scsi_defaults(default_info_type *default_info) {

   default_info->device_buffer_size =
             calculate_device_buffer_size((dcb_scsi_type *) default_info->dcb);
   if (default_info->device_buffer_size == 0) {
      default_info->recommended_buffers = 2;
      default_info->recommended_buffer_size = 32768;
      default_info->recommended_buffer_pool_size = 65536;
      default_info->recommended_write_buffer_full_threshold = 32768;
      default_info->recommended_read_buffer_empty_threshold = 32768;
   } else {
      default_info->recommended_buffers = 2;
      default_info->recommended_buffer_size =
                                 _min(default_info->device_buffer_size, 32768);
      default_info->recommended_buffer_pool_size =
                                    2 * default_info->recommended_buffer_size;
      default_info->recommended_write_buffer_full_threshold =
                                         default_info->recommended_buffer_size;
      default_info->recommended_read_buffer_empty_threshold =
                                         default_info->recommended_buffer_size;
   }
   return(ercOK);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When opening a SCSI sequential access device, first obtain either a file
 handle (older CTOS) or a path handle (CTOS with SCSI Manager) to use for
 subsequent commands issued to the device.  If this is OK, obtain the current
 status information (just to clear a possible UNIT ATTENTION), then reserve
 the unit (this is with an eye to future multiple-initiator environments), get
 inquiry data to see if we are dealing with a SCSI-2 device and lastly examine
 the mode pages to see how the device is configured (buffered vs. unbuffered
 and fixed vs. variable length records). */

unsigned scsi_open(dcb_scsi_type *dcb) {

   static char direct_io_name[4] = "[Z0]";
   unsigned erc;
   inquiry_data_type inquiry_data;
   seq_parameters_type seq_parameters;

   if (internal_os_version >= 0x0C00)
      erc = OpenScsiPath(&dcb->scsi_manager_name[1],
                         dcb->scsi_manager_name[0], dcb->host_adapter,
                         dcb->scsi_id, dcb->lun, &dcb->scsi_handle, NULL, 0,
                         NULL, 0, MODE_EXCLUSIVE);
   else {
      memcpy(&direct_io_name[1], dcb->direct_io_name,
             sizeof(dcb->direct_io_name));
      erc = OpenFile(&dcb->scsi_handle, direct_io_name, sizeof(direct_io_name),
                     dcb->direct_io_name, sizeof(dcb->direct_io_name), 
                     modeModify);
   }
   if (erc != ercOK)			/* Succeed in establishing a handle? */
      return(erc);
   scsi_status(dcb);			/* Clear possible UNIT ATTENTION */
   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->cdb_len = 6;
   dcb->cdb[0] = RESERVE_UNIT;		/* Attempt unit reservation... */
   dcb->data_phase = NO_DATA_PHASE;
   dcb->io_data = NULL;
   dcb->io_xfer_count = 0;
   issue_scsi_command(dcb, TRUE);	/* ...but ignore any errors */
   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->cdb[0] = INQUIRY;		/* Is the device SCSI or SCSI-2? */
   dcb->cdb[4] = sizeof(inquiry_data);
   dcb->data_phase = DATA_IN;
   dcb->io_data = &inquiry_data;
   dcb->io_xfer_count = sizeof(inquiry_data);
   if ((erc = issue_scsi_command(dcb, TRUE)) != ercOK)
      return(erc);
   dcb->operating_mode.scsi2 = (dcb->target_status == GOOD)
                    ? ((inquiry_data.device_versions & ANSI_VERSION_MASK) >= 2)
                    : FALSE;
   memset(&seq_parameters, 0, sizeof(seq_parameters));
   erc = scsi_mode_query(dcb, &seq_parameters);
   dcb->min_record_size = seq_parameters.min_record_size;
   dcb->max_record_size = seq_parameters.max_record_size;
   dcb->block_size = seq_parameters.block_size;
   return(erc);

}

/*-----------------------------------------------------------------------------
 At the close, release our earlier reservation and then relinquish the CTOS
 handle used to access the device.  If the device is BUSY, for example,
 rewinding, the RELEASE UNIT fails.  In the present CTOS single-initiator
 environment, this is not a problem but is something to be resolved later. */

unsigned scsi_close(dcb_scsi_type *dcb) {

   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->cdb_len = 6;
   dcb->cdb[0] = RELEASE_UNIT;		/* Attempt to release reservation... */
   dcb->data_phase = NO_DATA_PHASE;
   dcb->io_data = NULL;
   dcb->io_xfer_count = 0;
   issue_scsi_command(dcb, TRUE);	/* ...but ignore any errors */
   return((internal_os_version >= 0x0C00) ? CloseScsiPath(dcb->scsi_handle)
                                          : CloseFile(dcb->scsi_handle));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Issue various MODE SENSE, READ BLOCK LIMITS and READ BUFFER commands to the
 SCSI device in order to determine its current operating characteristics.  Not
 all SCSI devices that the Sequential Access service has to support have all
 of these commands implemented, so we must be fairly forgiving if an error
 occurs on one of them (i.e. we fabricate plausible data to return to the
 user!).  The generic SeqAccessModeQuery logic has already filled in portions
 of the mode parameters that it can determine from the Dcb, we need only
 concern ourselves with SCSI specific portions here.  The major problem in
 interpreting SCSI mode data is the LSB/MSB reversal between the SCSI standard
 and Intel 80X86 processors (see some of the tortured code below). */

unsigned scsi_mode_query(dcb_scsi_type *dcb,
                         seq_parameters_type *seq_parameters) {

   block_descriptor_type block_descriptor;
   unsigned erc;
   device_page_type device_page;
   error_recovery_page_type error_recovery_page;
   mode_header6_type mode_header;

   if ((erc = mode_sense(dcb, (dcb->operating_mode.scsi2) ? ALL_PAGES : 0,
                         CURRENT_MODE_VALUES, &mode_header, &block_descriptor,
                         NULL, 0)) != ercOK)
      return(erc);
   seq_parameters->write_protected = dcb->unit_status.write_protected =
                                     ((mode_header.device_specific & WP) != 0);
   dcb->operating_mode.buffered =
              (((mode_header.device_specific & BUFFERED_MODE_MASK) >> 4) != 0);
   seq_parameters->unbuffered = !dcb->operating_mode.buffered;
   seq_parameters->speed = mode_header.device_specific & SPEED_MASK;
   seq_parameters->density = dcb->density = block_descriptor.density_code;
   to_intel_byte_order((char *) block_descriptor.total_blocks,
                       (char *) &seq_parameters->total_blocks,
                       sizeof(block_descriptor.total_blocks));
   to_intel_byte_order((char *) block_descriptor.block_length,
                       (char *) &seq_parameters->block_size,
                       sizeof(block_descriptor.block_length));
   seq_parameters->variable_length = dcb->operating_mode.variable_length =
                         ((dcb->block_size = seq_parameters->block_size) == 0);
   if (dcb->min_record_size == 0 || dcb->max_record_size == 0)
      if (seq_parameters->variable_length)
         read_block_limits(dcb, &dcb->min_record_size, &dcb->max_record_size);
      else
         dcb->min_record_size = dcb->max_record_size = dcb->block_size;
   seq_parameters->min_record_size = dcb->min_record_size;
   seq_parameters->max_record_size = dcb->max_record_size;
   seq_parameters->device_buffer_size = calculate_device_buffer_size(dcb);
   if (mode_sense(dcb, ERROR_RECOVERY_PAGE, CURRENT_MODE_VALUES,
                  NULL, NULL, &error_recovery_page,
                  sizeof(error_recovery_page)) == ercOK) {
      seq_parameters->report_soft_errors =
                                ((error_recovery_page.error_flags & PER) != 0);
      seq_parameters->disable_error_correction =
                                ((error_recovery_page.error_flags & DCR) != 0);
      seq_parameters->disable_read_retries =
                                   (error_recovery_page.read_retry_limit == 0);
      seq_parameters->disable_write_retries =
                                  (error_recovery_page.write_retry_limit == 0);
      seq_parameters->read_retry_limit = error_recovery_page.read_retry_limit;
      seq_parameters->write_retry_limit =
                                         error_recovery_page.write_retry_limit;
   }
   if (mode_sense(dcb, DEVICE_PAGE, CURRENT_MODE_VALUES, NULL, NULL,
                  &device_page, sizeof(device_page)) == ercOK) {
      dcb->operating_mode.buffer_recoverable =
                                       ((device_page.device_flags & DBR) != 0);
      seq_parameters->data_buffer_nonrecoverable =
                                       !dcb->operating_mode.buffer_recoverable;
      seq_parameters->disable_automatic_velocity =
                                       ((device_page.device_flags & AVC) == 0);
      seq_parameters->buffer_recovery_LIFO =
                                       ((device_page.device_flags & RBO) != 0);
      seq_parameters->checkpoint_EOM =
                                      ((device_page.device_flags2 & SEW) != 0);
      seq_parameters->data_compression = dcb->operating_mode.data_compression
         = (device_page.data_compression != 0);
      seq_parameters->gap_size = device_page.gap_size;
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 When the operating mode for SCSI devices is established, we have to deal with
 SCSI-1 devices as well as SCSI-2 devices, and SCSI-2 devices which may not
 have all the mode pages implemented.  At a minimum we must be able to read
 and change the block descriptors associated with mode data, since the block
 descriptor indicates (or controls) buffered vs. unbuffered mode and variable
 length vs. fixed length records.  The other two pages of interest, the error
 recover page and the device parameters page, we will attempt to change if
 they are present and ignore otherwise.  Note that mode page values obtained
 from the device itself are used as the start of the altered data to be fed
 back to the device; unless the user has specified that SCSI pages are NOT to
 be reset to their default values, we read the default pages to obtain this
 information.  Also note that the "changeable" page is a bit mask by means of
 which the device communicates its permission as to which of the fields in the
 page may actually be updated by the user. */

unsigned scsi_mode_set(dcb_scsi_type *dcb,
                       seq_parameters_type *seq_parameters) {

   block_descriptor_type block_descriptor;
   unsigned long block_size, max_block_length;
   device_page_type device_page, device_page_changeable;
   unsigned erc, min_block_length;
   error_recovery_page_type error_recovery_page,
      error_recovery_page_changeable;
   mode_header6_type mode_header;
   char mode_values;

   read_block_limits(dcb, &min_block_length, &max_block_length);
   if (     (   seq_parameters->variable_length
             && (   seq_parameters->block_size != 0
                 || seq_parameters->min_record_size
                     >= seq_parameters->max_record_size
                 || seq_parameters->min_record_size < min_block_length
                 || seq_parameters->max_record_size > max_block_length))
         || (   !seq_parameters->variable_length
             && (   seq_parameters->min_record_size
                     != seq_parameters->max_record_size
                 || seq_parameters->min_record_size
                     > seq_parameters->block_size
                 || (   seq_parameters->min_record_size != 0
                     && seq_parameters->block_size
                         % seq_parameters->min_record_size != 0)
                 || (   seq_parameters->block_size != 0
                     && seq_parameters->block_size < min_block_length)
                 || seq_parameters->block_size > max_block_length)))
      return(ercInvalidTapeParams);
   mode_values = (   seq_parameters->suppress_default_mode_on_open
                  || !dcb->operating_mode.scsi2)
                                   ? CURRENT_MODE_VALUES : DEFAULT_MODE_VALUES;
   if ((erc = mode_sense(dcb, (dcb->operating_mode.scsi2) ? ALL_PAGES : 0,
                         mode_values, &mode_header, &block_descriptor,
                         NULL, 0)) != ercOK)
      return(erc);
   mode_header.device_specific =
             ((dcb->operating_mode.buffered) ? BUFFERED_MODE : UNBUFFERED_MODE)
             | seq_parameters->speed;
   block_descriptor.density_code = seq_parameters->density;
   if (seq_parameters->variable_length)
      memset(&block_descriptor.block_length, 0,
             sizeof(block_descriptor.block_length));
   else if (seq_parameters->block_size == 0) {
      block_size = dcb->block_size;
      to_scsi_byte_order((char *) &block_size,
                         (char *) block_descriptor.block_length,
                         sizeof(block_descriptor.block_length));
   } else
      to_scsi_byte_order((char *) &seq_parameters->block_size,
                         (char *) block_descriptor.block_length,
                         sizeof(block_descriptor.block_length));
   if (     (erc = mode_select(dcb, &mode_header, &block_descriptor,
                               NULL, 0)) != ercOK
         || (erc = mode_sense(dcb, (dcb->operating_mode.scsi2) ? ALL_PAGES : 0,
                              CURRENT_MODE_VALUES, &mode_header,
                              &block_descriptor, NULL, 0)) != ercOK)
      return(erc);
   dcb->operating_mode.buffered =
              (((mode_header.device_specific & BUFFERED_MODE_MASK) >> 4) != 0);
   dcb->density = block_descriptor.density_code;
   to_intel_byte_order((char *) block_descriptor.block_length,
                       (char *) &block_size,
                       sizeof(block_descriptor.block_length));
   dcb->operating_mode.variable_length = ((dcb->block_size = block_size) == 0);
   if (seq_parameters->min_record_size != 0)
      dcb->min_record_size = seq_parameters->min_record_size;
   if (seq_parameters->max_record_size != 0)
      dcb->max_record_size = seq_parameters->max_record_size;
   if (     mode_sense(dcb, ERROR_RECOVERY_PAGE, CHANGEABLE_MODE_VALUES,
                       NULL, NULL, &error_recovery_page_changeable,
                       sizeof(error_recovery_page_changeable)) == ercOK
         && mode_sense(dcb, ERROR_RECOVERY_PAGE, mode_values, NULL, NULL,
                       &error_recovery_page,
                       sizeof(error_recovery_page)) == ercOK) {
      if (error_recovery_page_changeable.error_flags & PER)
         if (seq_parameters->report_soft_errors)
            error_recovery_page.error_flags |= PER;
         else
            error_recovery_page.error_flags &= ~PER;
      if (error_recovery_page_changeable.error_flags & DCR)
         if (seq_parameters->disable_error_correction)
            error_recovery_page.error_flags |= DCR;
         else
            error_recovery_page.error_flags &= ~DCR;
      if (error_recovery_page_changeable.read_retry_limit != 0)
         if (seq_parameters->disable_read_retries)
            error_recovery_page.read_retry_limit = 0;
         else if (   seq_parameters->read_retry_limit != 0
                  && (   seq_parameters->read_retry_limit
                      <= error_recovery_page_changeable.read_retry_limit))
            error_recovery_page.read_retry_limit =
                                              seq_parameters->read_retry_limit;
      if (error_recovery_page_changeable.write_retry_limit != 0)
         if (seq_parameters->disable_write_retries)
            error_recovery_page.write_retry_limit = 0;
         else if (   seq_parameters->write_retry_limit != 0
                  && (   seq_parameters->write_retry_limit
                      <= error_recovery_page_changeable.write_retry_limit))
            error_recovery_page.write_retry_limit =
                                             seq_parameters->write_retry_limit;
      mode_select(dcb, &mode_header, NULL, &error_recovery_page,
                  sizeof(error_recovery_page));
   }
   if (     mode_sense(dcb, DEVICE_PAGE, CHANGEABLE_MODE_VALUES,
                       NULL, NULL, &device_page_changeable,
                       sizeof(device_page_changeable)) == ercOK
         && mode_sense(dcb, DEVICE_PAGE, mode_values, NULL, NULL, &device_page,
                       sizeof(device_page)) == ercOK) {
      if (device_page_changeable.device_flags & AVC) 
         if (seq_parameters->disable_automatic_velocity)
            device_page.device_flags &= ~AVC;
         else
            device_page.device_flags |= AVC;
      if (device_page_changeable.device_flags & RBO) 
         if (seq_parameters->buffer_recovery_LIFO)
            device_page.device_flags |= RBO;
         else
            device_page.device_flags &= ~RBO;
      if (seq_parameters->gap_size <= device_page_changeable.gap_size)
         device_page.gap_size = seq_parameters->gap_size;
      if (device_page_changeable.device_flags2 & SEW) 
         if (seq_parameters->checkpoint_EOM)
            device_page.device_flags2 |= SEW;
         else
            device_page.device_flags2 &= ~SEW;
      if (device_page_changeable.data_compression != 0)
         device_page.data_compression =
                                    (seq_parameters->data_compression) ? 1 : 0;
      mode_select(dcb, &mode_header, NULL, &device_page, sizeof(device_page));
      if (mode_sense(dcb, DEVICE_PAGE, CURRENT_MODE_VALUES, NULL, NULL,
                     &device_page, sizeof(device_page)) == ercOK) {
         dcb->operating_mode.buffer_recoverable =
                                       ((device_page.device_flags & DBR) != 0);
         dcb->operating_mode.data_compression =
                                           (device_page.data_compression != 0);
      }
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Obtain current status information from the SCSI device by a combbination of
 TEST UNIT READY and REQUEST SENSE.  It is OK to block the Sequential Access
 service for this operation since we do not want any intervening requests to
 muddy the picture of the status information that is to be returned.  Once the
 information has been assembled, map it to a set of "virtual" status
 information that is generic for all interface classes and is kept in the Dcb
 for the device.  Note that if this procedure was called because status was
 "pending" (i.e. a deferred error condition was detected while doing buffered
 reads or writes), the client has not yet had a chance to call SeqAccessStatus
 to obtain detailed information---so keep the sense data "fresh" for one more
 call, if the client so desires. */

unsigned scsi_status(dcb_scsi_type *dcb) {

   unsigned erc;
   mode_header6_type mode_header;

   if (dcb->state.status_available)
      erc = map_scsi_status(dcb, ercOK);
   else {
      memset(dcb->cdb, 0, sizeof(dcb->cdb));
      dcb->cdb_len = 6;
      dcb->timeout = 15;
      dcb->data_phase = NO_DATA_PHASE;
      dcb->io_data = NULL;
      dcb->io_xfer_count = 0;
      if (     (erc = issue_scsi_command(dcb, TRUE)) == ercOK
            && dcb->target_status == GOOD) {	/* May need more details... */
         dcb->cdb[0] = REQUEST_SENSE;
         dcb->cdb[4] = sizeof(dcb->ext_sense);
         dcb->data_phase = DATA_IN;
         dcb->io_data = dcb->ext_sense;
         dcb->io_xfer_count = sizeof(dcb->ext_sense);
         if (     (erc = issue_scsi_command(dcb, TRUE)) == ercOK
               && dcb->target_status == GOOD)
            dcb->target_status = CHECK_CONDITION;	/* Trickery! */
      }
      erc = (erc == ercOK) ? map_scsi_status(dcb, ercOK)
                           : ercTapeStatusUnavailable;
      if (mode_sense(dcb, (dcb->operating_mode.scsi2) ? ALL_PAGES : 0,
                     CURRENT_MODE_VALUES, &mode_header,
                     NULL, NULL, 0) == ercOK)
         dcb->unit_status.write_protected =
                                     ((mode_header.device_specific & WP) != 0);
   }
   if (!dcb->state.pending_status)		/* If it's not a deferred... */
      dcb->state.status_available = FALSE;	/* ...error, clear status */
   dcb->state.pending_status = FALSE;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 For buffered data transfer operations, build either a READ or a WRITE command
 descriptor with the correct transfer length specification and fixed/variable
 code.  Set a nominal 40 second timeout and let another function determine
 whether or not to use the SCSI Manager to issue the command to the device.
 So that the Sequential Access service does not block on potentially lengthy
 data transfer operations, request completion procedures occur elsewhere after
 a response is made. */

unsigned scsi_io(dcb_scsi_type *dcb) {

   unsigned block_count;

   dcb->timeout = 40;
   dcb->io_data = dcb->device_buffer->data;
   dcb->io_xfer_count = dcb->device_buffer->available;
   dcb->data_phase = (dcb->state.inbound) ? DATA_IN : DATA_OUT;
   dcb->cdb_len = 6;
   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->cdb[0] = (dcb->data_phase == DATA_IN) ? READ : WRITE;
   if (dcb->operating_mode.variable_length) {
      dcb->cdb[4] = dcb->io_xfer_count;
      dcb->cdb[3] = dcb->io_xfer_count >> 8;
   } else if (   dcb->min_record_size == dcb->block_size
              || !dcb->operating_mode.scsi2) {
      if (dcb->io_xfer_count < dcb->block_size)	/* Can only happen on WRITE */
         dcb->device_buffer->available = dcb->io_xfer_count = dcb->block_size;
      dcb->cdb[1] = FIXED;
      block_count = (dcb->io_xfer_count / dcb->block_size);
      dcb->io_xfer_count = block_count * dcb->block_size;
      dcb->cdb[4] = block_count;
      dcb->cdb[3] = block_count >> 8;
   } else if (dcb->cdb[0] == READ) {		/* Use variable reads... */
      if (internal_os_version >= 0x0C00)
         dcb->cdb[1] = SILI;
      dcb->cdb[4] = dcb->io_xfer_count;
      dcb->cdb[3] = dcb->io_xfer_count >> 8;
   } else if (dcb->io_xfer_count < dcb->block_size) {
      dcb->cdb[4] = dcb->io_xfer_count;	/* Variable write for short block */
      dcb->cdb[3] = dcb->io_xfer_count >> 8;
   } else {
      dcb->cdb[1] = FIXED;
      block_count = (dcb->io_xfer_count / dcb->block_size);
      dcb->io_xfer_count = block_count * dcb->block_size;
      dcb->cdb[4] = block_count;
      dcb->cdb[3] = block_count >> 8;
   }
   return(issue_scsi_command(dcb, FALSE));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Operations requested by the user by the SeqAccessControl interface are
 directed to this function.  Decode the control function and its qualifier
 into a SCSI command descriptor block, determine an appropriate timeout and
 then issue the command to the device.  Any interpretation of the result of
 the command is performed elsewhere (when a response is made). */

unsigned scsi_ctrl(dcb_scsi_type *dcb) {

   seq_access_rq_type *rq;

   dcb->cdb_len = 6;
   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->data_phase = NO_DATA_PHASE;
   dcb->io_data = NULL;
   dcb->io_xfer_count = 0;
   rq = dcb->rq;
   switch (rq->ctrl_function) {
      case CTRL_REWIND:
         dcb->cdb[0] = REWIND;
         if (rq->ctrl_qualifier)
            dcb->cdb[1] |= IMMED;
         dcb->timeout = 300;
         break;

      case CTRL_UNLOAD:
         dcb->cdb[0] = LOAD_UNLOAD;
         if (rq->ctrl_qualifier)
            dcb->cdb[1] |= IMMED;
         dcb->timeout = 300;
         break;

      case CTRL_RETENSION:
         dcb->cdb[0] = LOAD_UNLOAD;
         if (rq->ctrl_qualifier)
            dcb->cdb[1] |= IMMED;
         dcb->cdb[4] |= (RE_TEN | LOAD);
         dcb->timeout = 300;
         break;

      case CTRL_ERASE_MEDIUM:
         dcb->cdb[0] = ERASE;
         if (rq->ctrl_qualifier)
            dcb->cdb[1] |= IMMED_ERASE;
         dcb->cdb[1] |= LONG;
         dcb->timeout = (dcb->density == DDS) ? 10800 : 1500;
         break;

      case CTRL_WRITE_FILEMARK:
         dcb->cdb[0] = WRITE_FILEMARKS;
         dcb->cdb[3] = rq->ctrl_qualifier >> 8;
         dcb->cdb[4] = rq->ctrl_qualifier & 0x00FF;
         dcb->timeout = 60;
         break;

      case CTRL_SCAN_FILEMARK:
         dcb->cdb[0] = SPACE;
         dcb->cdb[1] |= SPACE_SEQ_FILEMARKS;
         if (rq->ctrl_qualifier < 0)
            dcb->cdb[2] = 0xFF;
         dcb->cdb[3] = rq->ctrl_qualifier >> 8;
         dcb->cdb[4] = rq->ctrl_qualifier & 0x00FF;
         dcb->timeout = 1500;
         break;

      case CTRL_SPACE_RECORD:
         dcb->cdb[0] = SPACE;
         dcb->cdb[1] |= SPACE_BLOCKS;
         if (rq->ctrl_qualifier < 0)
            dcb->cdb[2] = 0xFF;
         dcb->cdb[3] = rq->ctrl_qualifier >> 8;
         dcb->cdb[4] = rq->ctrl_qualifier & 0x00FF;
         dcb->timeout = 1500;
         break;

      case CTRL_ERASE_GAP:
         dcb->cdb[0] = ERASE;
         dcb->timeout = 60;
         break;

      default:
         return(ercTapeCommandIllegal);
   }
   return(issue_scsi_command(dcb, FALSE));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Recover data still held in the device's buffers.  This is invoked by
 SeqAccessRecoverBufferData, most usually at the end of a tape when
 early-warning EOM has been returned by the device. */

unsigned scsi_buffer(dcb_scsi_type *dcb) {

   dcb->timeout =15;
   dcb->io_data = dcb->rq_data;
   dcb->io_xfer_count = dcb->rq_residual;
   dcb->data_phase = DATA_IN;
   dcb->cdb_len = 6;
   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->cdb[0] = RECOVER_BUFFERED_DATA;
   if (dcb->operating_mode.variable_length) {
      dcb->cdb[4] = dcb->io_xfer_count;
      dcb->cdb[3] = dcb->io_xfer_count >> 8;
   } else {
      dcb->cdb[1] |= FIXED;
      dcb->cdb[4] = (dcb->io_xfer_count / dcb->block_size);
      dcb->cdb[3] = (dcb->io_xfer_count / dcb->block_size) >> 8;
   }
   return(issue_scsi_command(dcb, FALSE));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 SCSI requests for which the Sequential Access service does not block (i.e
 does not wait for a response on our default response exchange) eventually
 arrive at the service exchange and are processed here.  Take into account the
 state of the Dcb---particularly what command was last executed---in order to
 call the appropriate function to interpret information returned from the just
 completed operation.  Also note that if SCSI Manager calls are NOT being
 used, the autosense capability of the SCSI Manager must be simulated with a
 separate DoDirectRead operation that specifies REQUEST SENSE.  When this
 operation completes, it too is returned to this procedure. */

unsigned scsi_rq_done(dcb_scsi_type *dcb) {

   unsigned erc;
   signed long residual = 0;

   if (internal_os_version < 0x0C00) {
      if (dcb->state.autosense)
         erc = (dcb->autosense_status == GOOD) ?
                                 dcb->own_rq.erc_ret : ercSenseDataUnavailable;
      else if ((erc = dcb->own_rq.erc_ret) == ercOK)
         if (dcb->target_status == GOOD)
            dcb->io_xfer_actual = dcb->io_xfer_count;
         else if (   dcb->target_status == CHECK_CONDITION
                  && (erc = request_sense(dcb)) == ercOK)
            return(REQUEST_IN_PROGRESS);
   } else
      erc = dcb->own_rq.erc_ret;
   if (     erc == ercOK
         && dcb->target_status == CHECK_CONDITION
         && (dcb->ext_sense[0] & INFO_VALID)) {
      to_intel_byte_order((char *) &dcb->ext_sense[3],
                          (char *) &residual, sizeof(residual));
      if (dcb->state.data_transfer) {
         if (dcb->cdb[1] & FIXED)
            residual *= dcb->block_size;
         if (residual < 0) {
            dcb->io_xfer_actual = dcb->io_xfer_count;
            dcb->residual = residual;
         } else if (residual <= dcb->io_xfer_count) {
            dcb->io_xfer_actual = dcb->io_xfer_count - residual;
            dcb->residual = 0;
         } else {
            dcb->io_xfer_actual = 0;
            dcb->residual = residual - dcb->io_xfer_count;
         }
      }
   }
   if (erc != ercOK || dcb->target_status != GOOD) {
      dcb->state.status_available = TRUE;
      dcb->state.pending_status = ((erc = map_scsi_status(dcb, erc)) != ercOK);
   }
   if (     dcb->cdb[0] == WRITE_FILEMARKS
         && residual != 0
         && erc == ercTapeEomWarning) {
      dcb->cdb[3] = residual >> 8;
      dcb->cdb[4] = residual;
      return(issue_scsi_command(dcb, FALSE));
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 SCSI commands are built in the Dcb (along with the appropriate pointers to
 data buffers) before this procedure is called.  Three functions are served
 here.  First, distinguish between CTOS prior to 3.0 (which must use
 DoDirectRead and DoDirectWrite to issue SCSI commands) and the later CTOS
 versions which include the SCSI Manager (and allow the use of ScsiCdbDataIn
 and ScsiCdbDataOut with their autosense capabilities).  Second, after issuing
 the request either wait for the response or return immediately to the caller
 with an "in progress" indication---this is determined by the
 'wait_for_response' variable passed as a parameter.  Last, if the caller has
 indicated a wait and DoDirectRead/DoDirectWrite are in use, see if
 CHECK CONDITION has been returned by the target and simulate autosense
 capabilities to retrieve the extended sense data if this is the case.  Note
 that a different procedure, 'scsi_rq_done', performs this function if the
 caller requested and immediate return. */

private unsigned issue_scsi_command(dcb_scsi_type *dcb,
                                    Boolean wait_for_response) {

   direct_rq_type *direct_rq;
   unsigned erc;
   void *response_rq;
   scsi_rq_type *scsi_rq;

   dcb->state.status_available = FALSE;	/* Sense data cleared by new command */
   dcb->io_xfer_actual = 0;
   if (internal_os_version >= 0x0C00) {	/* CTOS 3.0 first for SCSI Manager */
      dcb->target_status = 0x00FF;
      memset((scsi_rq = (scsi_rq_type *) &dcb->own_rq), 0,
             sizeof(dcb->own_rq));
      scsi_rq->s_cnt_info = 6;
      if (dcb->data_phase == DATA_OUT) {
         scsi_rq->n_req_pb_cb = 2;
         scsi_rq->n_resp_pb_cb = 3;
         scsi_rq->rq_code = SCSI_CDB_DATA_OUT_RQ_CODE;
      } else {
         scsi_rq->n_req_pb_cb = 1;
         scsi_rq->n_resp_pb_cb = 4;
         scsi_rq->rq_code = SCSI_CDB_DATA_IN_RQ_CODE;
      }
      scsi_rq->exch_resp = (wait_for_response) ? default_resp_exch : serv_exch;
      scsi_rq->path_handle = dcb->scsi_handle;
      scsi_rq->timeout = dcb->timeout * 10;
      scsi_rq->cdb = dcb->cdb;
      scsi_rq->cdb_len = dcb->cdb_len;
      scsi_rq->data = dcb->io_data;
      scsi_rq->data_len = dcb->io_xfer_count;
      scsi_rq->xfer_count = &dcb->io_xfer_actual;
      scsi_rq->size_xfer_count = sizeof(dcb->io_xfer_actual);
      scsi_rq->target_status = &dcb->target_status;
      scsi_rq->size_target_status = 1;
      scsi_rq->ext_sense = dcb->ext_sense;
      scsi_rq->ext_sense_len = sizeof(dcb->ext_sense);
   } else {				/* Otherwise, use direct I/O */
      dcb->target_status = dcb->autosense_status = 0xFFFF;
      dcb->state.autosense = FALSE;
      memset(dcb->ext_sense, 0, sizeof(dcb->ext_sense));
      memset(direct_rq = (direct_rq_type *) &dcb->own_rq, 0,
             sizeof(dcb->own_rq));
      direct_rq->s_cnt_info = 6;
      if (dcb->data_phase == DATA_IN) {
         direct_rq->n_req_pb_cb = 1;
         direct_rq->n_resp_pb_cb = 2;
         direct_rq->rq_code = DO_DIRECT_READ_RQ_CODE;
      } else {
         direct_rq->n_req_pb_cb = 2;
         direct_rq->n_resp_pb_cb = 1;
         direct_rq->rq_code = DO_DIRECT_WRITE_RQ_CODE;
      }
      direct_rq->exch_resp = (wait_for_response) ?
                                                 default_resp_exch : serv_exch;
      direct_rq->file_handle = dcb->scsi_handle;
      direct_rq->options = (dcb->state.data_transfer) ? DIRECT_IO_DMA : 0;
      direct_rq->timeout = dcb->timeout / DIRECT_IO_TIME_UNIT;
      direct_rq->cdb = dcb->cdb;
      direct_rq->cdb_len = dcb->cdb_len;
      direct_rq->data = dcb->io_data;
      direct_rq->data_len = dcb->io_xfer_count;
      direct_rq->result = &dcb->target_status;
      direct_rq->size_result = sizeof(dcb->target_status);
   }
   if ((erc = Request(&dcb->own_rq)) == ercOK)
      if (wait_for_response)
         do
            if ((erc = Wait(default_resp_exch, &response_rq)) == ercOK)
               if (response_rq == &dcb->own_rq)
                  if (internal_os_version >= 0x0C00)
                     return(dcb->own_rq.erc_ret);
                  else if (dcb->state.autosense)
                     return((dcb->autosense_status == GOOD) ?
                                dcb->own_rq.erc_ret : ercSenseDataUnavailable);
                  else if (dcb->target_status == CHECK_CONDITION)
                     erc = request_sense(dcb);
                  else
                     return(dcb->own_rq.erc_ret);
         while (erc == ercOK);
      else
         erc = REQUEST_IN_PROGRESS;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Because CTOS prior to 3.0 does not include a SCSI Manager that provides
 automatic retrieval of sense data after a CHECK CONDITION, this "autosense"
 capability must be simulated with an independent DoDirectRead operation that
 specifies REQUEST SENSE.  This function is called from different locations,
 dependant upon whether or not the Sequential Access service is to block or
 not block when issuing a SCSI operation. */

private unsigned request_sense(dcb_scsi_type *dcb) {

   direct_rq_type *direct_rq;

   dcb->state.autosense = TRUE;
   direct_rq = (direct_rq_type *) &dcb->own_rq;
   direct_rq->n_req_pb_cb = 1;
   direct_rq->n_resp_pb_cb = 2;
   direct_rq->rq_code = DO_DIRECT_READ_RQ_CODE;
   direct_rq->timeout = 15 / DIRECT_IO_TIME_UNIT;
   direct_rq->cdb = request_sense_cdb;
   direct_rq->cdb_len = sizeof(request_sense_cdb);
   direct_rq->data = dcb->ext_sense;
   direct_rq->data_len = sizeof(dcb->ext_sense);
   direct_rq->result = &dcb->autosense_status;
   direct_rq->size_result = sizeof(dcb->autosense_status);
   return(Request(direct_rq));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Common procedure to assemble a proper SCSI MODE SELECT command from the mode
 header, optional block descriptor(s) and the mode page data itself.  Because
 this procedure is used by code that uses a subsequent MODE SENSE to
 double-check the actual results of the MODE SELECT it is not critical that we
 obtain sense data if a CHECK CONDITION resulted from the command. */

private unsigned mode_select(dcb_scsi_type *dcb,
                             mode_header6_type *mode_header,
                             block_descriptor_type *block_descriptor,
                             void *page, char page_len) {

   char mode_data[255];

   memcpy(dcb->io_data= mode_data, mode_header, sizeof(*mode_header));
   mode_data[0] = 0;
   if (block_descriptor != NULL) {
      memcpy(&mode_data[sizeof(*mode_header)], block_descriptor,
             mode_data[3] = sizeof(*block_descriptor));
      memcpy(&mode_data[sizeof(*mode_header) + sizeof(*block_descriptor)],
             page, page_len);
      dcb->io_xfer_count = sizeof(*mode_header) + sizeof(*block_descriptor)
                            + page_len;
   } else {
      mode_data[3] = 0;
      memcpy(&mode_data[sizeof(*mode_header)], page, page_len);
      dcb->io_xfer_count = sizeof(*mode_header) + page_len;
   }
   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->data_phase = DATA_OUT;
   dcb->cdb_len = 6;
   dcb->cdb[0] = MODE_SELECT;
   if (dcb->operating_mode.scsi2)
      dcb->cdb[1] = 0x10;
   dcb->cdb[4] = dcb->io_xfer_count;
   return(map_scsi_status(dcb, issue_scsi_command(dcb, TRUE)));

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Another shared procedure to obtain a SCSI mode page from the device and
 dissect it into three components: the mode header, the block descriptor(s)
 and the page data itself.  If the caller is not interested in any one of
 these structures, it is sufficient to supply NULL pointers for the structures
 in question. */

private unsigned mode_sense(dcb_scsi_type *dcb, char page_code,
                            char page_control, mode_header6_type *mode_header,
                            block_descriptor_type *block_descriptor,
                            void *page, char page_len) {

   unsigned erc;
   char mode_data[255];

   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->data_phase = DATA_IN;
   dcb->cdb_len = 6;
   dcb->cdb[0] = MODE_SENSE;
   if (block_descriptor == NULL && dcb->operating_mode.scsi2)
      dcb->cdb[1] = DBD;	/* Disable return of block descriptor(s) */
   dcb->cdb[2] =   (page_control & PAGE_CONTROL_MASK)
                 | (page_code & PAGE_CODE_MASK);
   memset(dcb->io_data = mode_data, 0,
          dcb->io_xfer_count = dcb->cdb[4] = sizeof(mode_data));
   if (     (erc = issue_scsi_command(dcb, TRUE)) == ercOK
         && dcb->target_status == GOOD) {
      if (mode_header == NULL)
         mode_header = (mode_header6_type *) mode_data;
      else
         memcpy(mode_header, mode_data, sizeof(*mode_header));
      if (mode_header->mode_data_length + 1 >= sizeof(*mode_header)) {
         if (block_descriptor != NULL)
            if (     mode_header->mode_data_length + 1 - sizeof(*mode_header)
                  >= sizeof(*block_descriptor))
               memcpy(block_descriptor, &mode_data[sizeof(*mode_header)],
                      sizeof(*block_descriptor));
            else
               memset(block_descriptor, 0, sizeof(*block_descriptor));
         if (page != NULL)
            if (     mode_header->mode_data_length + 1 - sizeof(*mode_header)
                      - mode_header->block_descriptor_length
                  >= page_len)
               memcpy(page, &mode_data[sizeof(mode_header6_type)
                                       + mode_header->block_descriptor_length],
                      page_len);
            else
               erc = ercScsiPageNonexistent;
      }
   } else if (erc == ercOK)
      erc = ercScsiPageNonexistent;
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 In order for users to make intelligent decisions as to how much buffer space
 to reserve for the recovery of data at early-warning EOM conditions (via the
 SeqAccessRecoverBufferData operation), the service must determine the total
 size of the device's buffers.  There is no one SCSI command that can provide
 this information, so for all 256 possible buffers we issue a READ BUFFER and
 just keep summing the size of the buffer description returned. */

private unsigned long calculate_device_buffer_size(dcb_scsi_type *dcb) {

   unsigned long buffer_capacity, total_buffer_capacity = 0;
   device_buffer_descriptor_type buffer_descriptor;
   unsigned i;

   if (dcb->operating_mode.scsi2) {
      memset(&dcb->cdb, 0, sizeof(dcb->cdb));
      dcb->data_phase = DATA_IN;
      dcb->cdb_len = 10;
      dcb->cdb[0] = READ_BUFFER;
      dcb->cdb[1] = BUFFER_DESCRIPTOR;
      dcb->cdb[8] = sizeof(buffer_descriptor);
      dcb->io_data = &buffer_descriptor;
      dcb->io_xfer_count = sizeof(buffer_descriptor);
      for (i = 0; i <= 0xFF; i++) {
         dcb->cdb[2] = i;
         if (     issue_scsi_command(dcb, TRUE) == ercOK
               && dcb->target_status == GOOD) {
            to_intel_byte_order((char *) buffer_descriptor.buffer_capacity,
                                (char *) &buffer_capacity,
                                sizeof(buffer_descriptor.buffer_capacity));
            total_buffer_capacity += buffer_capacity;
         }
      }
   }
   return(total_buffer_capacity);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Save a little code space.  This is invoked twice to return the devices
 minimum an maximum block size capabilities.  For fixed length records, the
 block length chosen by the user will have to fall in between these limits
 while for variable length records each of the minimum record (block) and
 maximum record (block) sizes must be compatible with the device's limits. */

private void read_block_limits(dcb_scsi_type *dcb, unsigned *min_block_length,
                          unsigned long *max_block_length) {

   block_limits_type block_limits;

   memset(&dcb->cdb, 0, sizeof(dcb->cdb));
   dcb->data_phase = DATA_IN;
   dcb->cdb_len = 6;
   dcb->cdb[0] = READ_BLOCK_LIMITS;
   dcb->io_data = &block_limits;
   dcb->io_xfer_count = sizeof(block_limits);
   if (     issue_scsi_command(dcb, TRUE) == ercOK
         && dcb->target_status == GOOD) {
      to_intel_byte_order((char *) block_limits.max_block_length,
                          (char *) max_block_length,
                          sizeof(block_limits.max_block_length));
      to_intel_byte_order((char *) &block_limits.min_block_length,
                          (char *) min_block_length,
                          sizeof(block_limits.min_block_length));
      if (     !dcb->operating_mode.scsi2
            && *min_block_length == *max_block_length
            && (   dcb->density == NRZI_800
                || dcb->density == PE_1600
                || dcb->density == PE_3200
                || dcb->density == GCR_6250)) {
         *min_block_length = 1;
         *max_block_length = 0xFFFF;
      }
   } else {
      *min_block_length = 1;
      *max_block_length = 0xFFFF;
   }

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 This function is called when a SCSI operation issued to the target has just
 completed.  If there was a CHECK CONDITION, an attempt has already been made
 to retrieve extended sense data and the 'autosense_status' field in the Dcb
 reflects whether or not sense data is present.  Take the specific SCSI sense
 data and map it into the "generic" sequential access device status
 information kept in the Dcb.  Also map the same information into a sequential
 access error return code (if one is available).  Note the order of precedence
 among EOM, filemark and ILI:  early-warning EOM has lowest precedence and the
 error code it generates may be usurped by either of filemark or ILI (which
 are supposed to be mutually exclusive!)*/

private unsigned map_scsi_status(dcb_scsi_type *dcb, unsigned erc) {

   if (erc == ercOK) {
      dcb->unit_status.on_line = TRUE;
      switch (dcb->target_status) {
         case GOOD:
            dcb->unit_status.ready = TRUE;
            dcb->unit_status.busy = FALSE;
            break;

         case TARGET_BUSY:
            erc = ercTapeBusy;
            dcb->unit_status.ready = FALSE;
            dcb->unit_status.busy = TRUE;
            break;

         case CHECK_CONDITION:
            if (dcb->ext_sense[2] & END_OF_MEDIUM) {
               if (     (dcb->operating_mode.scsi2 || !dcb->state.inbound)
                     && !dcb->state.ignore_EOM)
                  erc = ercTapeEomWarning;
               dcb->position.end_of_medium = TRUE;
            } else
               dcb->position.end_of_medium = FALSE;
            if (dcb->ext_sense[2] & FILE_MARK) {
               erc = ercTapeFileMark;
               dcb->command_status.file_mark = TRUE;
            } else
               dcb->command_status.file_mark = FALSE;
            if (     dcb->ext_sense[2] & ILI
                  && (   (dcb->cdb[1] & FIXED) != 0
                      || (   dcb->operating_mode.variable_length
                          && dcb->residual != 0)
                      || (   !dcb->operating_mode.variable_length
                          && dcb->io_xfer_actual % dcb->min_record_size
                              != 0))) {
               erc = ercInvalidTapeRecordSize;
               dcb->command_status.illegal_length = TRUE;
            } else
               dcb->command_status.illegal_length = FALSE;
            switch (dcb->ext_sense[2] & 0x0F) {
               case RECOVERED_ERROR:
                  dcb->command_status.recovered_error = TRUE;
                  break;

               case NOT_READY:
                  erc = ercTapeNotReady;
                  dcb->unit_status.ready = dcb->unit_status.busy = FALSE;
                  break;

               case MEDIUM_ERROR:
                  erc = ercTapeIoError;
                  dcb->command_status.hard_error = TRUE;
                  break;

               case HARDWARE_ERROR:
                  erc = ercTapeHardwareError;
                  dcb->command_status.hard_error = TRUE;
                  break;

               case ILLEGAL_REQUEST:
                  erc = ercTapeCommandIllegal;
                  dcb->command_status.illegal_operation = TRUE;
                  break;

               case UNIT_ATTENTION:
                  erc = ercTapeReset;
                  dcb->unit_status.reset = TRUE;
                  break;

               case DATA_PROTECT:
                  erc = ercTapeWriteProtected;
                  dcb->unit_status.write_protected = TRUE;
                  break;

               case BLANK_CHECK:
                  erc = ercTapeBlank;
                  dcb->position.end_of_data = TRUE;
                  break;

               case ABORTED_COMMAND:
                  erc = ercTapeCommandAborted;
                  dcb->command_status.hard_error = TRUE;
                  break;

               case VOLUME_OVERFLOW:
                  erc = ercTapeOverflow;
                  dcb->position.end_of_medium = TRUE;
                  break;
            }
            break;
      }
   } else if (erc == ercScsiTargetBusy) {	/* Prior command executing */
      erc = ercTapeBusy;
      dcb->unit_status.ready = FALSE;
      dcb->unit_status.busy = TRUE;
   } else if (erc == ercScsiSelectTimeout) {	/* The device is offline */
      erc = ercTapeNotReady;	/* Not a "real" error, just no status! */
      memset(dcb->ext_sense, 0, sizeof(dcb->ext_sense));
      dcb->unit_status.on_line =
         dcb->unit_status.ready =
         dcb->unit_status.busy =
         dcb->unit_status.write_protected = FALSE;
      dcb->position.beginning_of_medium =
         dcb->position.beginning_of_partition =
         dcb->position.end_of_data =
         dcb->position.end_of_partition =
         dcb->position.end_of_medium = FALSE;
      dcb->command_status.file_mark =
         dcb->command_status.illegal_length =
         dcb->command_status.illegal_operation =
         dcb->command_status.recovered_error =
         dcb->command_status.hard_error = FALSE;
   } else {			/* Some SCSI erc; invent best status we can */
      dcb->unit_status.ready = dcb->unit_status.busy = FALSE;
      dcb->position.beginning_of_medium =
         dcb->position.beginning_of_partition =
         dcb->position.end_of_data =
         dcb->position.end_of_partition =
         dcb->position.end_of_medium = FALSE;
   }
   return(erc);

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 The SCSI standard and Intel microprocessors see the world from opposite
 viewpoints:  what is the least significant byte of a binary number in SCSI
 data is the most significant byte to an 80X86 processor and vice-versa.  So,
 we need two trivial routines to translate back and forth.  Some numbers in
 SCSI data contain an odd number of bytes but all destination variables in
 this program are either two or four bytes long, so sometimes it is necessary
 to zero the uppermost byte of the destination. */

private void to_intel_byte_order(char *source, char *destination,
                                 unsigned source_len) {

   if (source_len & 0x0001)		/* Need to zero fill the MSB? */
      destination[source_len] = 0;
   destination += source_len - 1;
   do
      *destination-- = *source++;
   while (--source_len > 0);

}

private void to_scsi_byte_order(char *source, char *destination,
                                unsigned destination_len) {

   destination += destination_len - 1;
   do
      *destination-- = *source++;
   while (--destination_len > 0);

}
