/*****************************  MODULE HEADER  *******************************/
/*                                                                           */
/*                                                                           */
/*  MACHINE:                LANGUAGE:  Metaware C            OS: CTOS        */
/*                                                                           */
/*  install_seq_service.c                                                    */
/*                                                                           */
/*  Install/Deinstall front-end for CTOS Sequential Access service.          */
/*                                                                           */
/*  HISTORY:                                                                 */
/*  --------                                                                 */
/*                                                                           */
/*  MM/DD/YY  VVVV/MM  PROGRAMMER    /  DESCRIPTION                          */
/*                                                                           */
/*  03/27/91  121H.02  P. Johansson  /  Null case means "install", too.      */
/*  02/13/91  121G.01  C.G. Naik     /  Case sensitivity for FSrp.           */
/*  12/09/90  121E.00  P. Johansson  /  Created.                             */
/*                                                                           */
/*                    PROPRIETARY  PROGRAM  MATERIAL                         */
/*                                                                           */
/*  THIS MATERIAL IS PROPRIETARY TO UNISYS CORPORATION AND IS NOT TO         */
/*  BE REPRODUCED, USED OR DISCLOSED EXCEPT IN ACCORDANCE WITH PROGRAM       */
/*  LICENSE OR UPON WRITTEN AUTHORIZATION OF THE PATENT DIVISION OF          */
/*  UNISYS CORPORATION, DETROIT, MICHIGAN 48232, USA.                        */
/*                                                                           */
/*  COPYRIGHT (C) 1990 UNISYS CORPORATION. ALL RIGHTS RESERVED               */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*  UNISYS BELIEVES THAT THE SOFTWARE FURNISHED HEREWITH IS ACCURATE         */
/*  AND RELIABLE, AND MUCH CARE HAS BEEN TAKEN IN ITS PREPARATION. HOWEVER,  */
/*  NO RESPONSIBILITY, FINANCIAL OR OTHERWISE, CAN BE ACCEPTED FOR ANY       */
/*  CONSEQUENCES ARISING OUT OF THE USE OF THIS MATERIAL, INCLUDING LOSS     */
/*  OF PROFIT, INDIRECT, SPECIAL, OR CONSEQUENTIAL DAMAGES, THERE ARE NO     */
/*  WARRANTIES WHICH EXTEND BEYOND THE PROGRAM SPECIFICATION.                */
/*                                                                           */
/*  THE CUSTOMER SHOULD EXERCISE CARE TO ASSURE THAT USE OF THE SOFTWARE     */
/*  WILL BE IN FULL COMPLIANCE WITH LAWS, RULES AND REGULATIONS OF THE       */
/*  JURISDICTIONS WITH RESPECT TO WHICH IT IS USED.                          */
/*                                                                           */
/**************************  END OF MODULE HEADER  ***************************/

#ifdef debug
#define private
#else
#define private static
#endif

/* Standard C library macros and functions invoked by this module */

pragma Off(List);
#include <intel80X86.h>
#include <stdlib.h>
#include <string.h>
pragma Pop(List);

/* There are no procedures in the Sequential Access service that can cope with
   a variable number of arguments, so this pragma makes everything much more
   efficient.  However, it has to be established AFTER any standard C library
   functions are defined because it reverses the normal C convention. */

pragma Calling_convention(_CALLEE_POPS_STACK);

/* External CTOS and CTOS Toolkit functions invoked by this module */

#define BuildFileSpec
#define Chain
#define	CheckErc
#define CSubparams
#define ErrorExit
#define Exit
#define FSrp
#define GetPStructure
#define ParseFileSpec
#define QueryDefaultRespExch
#define RemovePartition
#define Request
#define RgParam
#define RgParamSetSimple
#define ULCmpB
#define VacatePartition
#define Wait

pragma Off(List);
#include <ctoslib.h>
pragma Pop(List);

#if defined(debug) && defined(breakpoint)
#undef breakpoint
extern void breakpoint(unsigned debug_value_for_AX);
#endif

/* Suppress C run-time (only CTOS functionality needed)

#include <stub.h>

/* Type definitions used by this module */

#define last(array) (sizeof(array) / sizeof(*array) - 1)

#define GetpStructureCase
#define sdType
#define Syscom
#define SysConfigType

#include <ctosTypes.h>
#include <ext_ctos_types.h>
#include "seq_service.h"

/* Other external functions in this application invoked by this module */

extern void exit_with_msg(unsigned erc, unsigned msg_num);
extern void initialize_msgs(void);
extern void log_msg(unsigned nls_msg_index, sdType nls_parms[],
                    unsigned nls_parms_len);
extern sdType *standard_msg(unsigned msg_num);

/* Error return codes used by this module */

#define FsErc
#define MpErc
#define RqErc
#define TapeErc

pragma Off(List);
#include <erc.h>
pragma Pop(List);

/* Static variables global within this manuscript */

/* Function prototypes defined before the functions themselves are declared */

void deinstall(void);
void install(void);

pragma Page(1);
/*-----------------------------------------------------------------------------
 Simply distinguish between the "install" and the "deinstall" cases.  Then
 exit. */

void main(void) {

   extern char sbVerRun[];
   sdType nls_parms[] = {         NULL, 0,
                         (void *) &sbVerRun[1], sbVerRun[0]};
   sdType sd_case;

   RgParam(0, 0, &nls_parms[0]);
   if (nls_parms[0].pb != NULL && nls_parms[0].cb != 0)
      log_msg(NLS_COMMAND_VERSION, nls_parms, sizeof(nls_parms));
   initialize_msgs();
   CheckErc(RgParam(0, 1, &sd_case));
   if (sd_case.cb == 0)
      install();
   else if (sd_case.cb != 2)
      exit_with_msg(ercSyntax, NLS_CASE_INVALID);
   else if (ULCmpB(sd_case.pb, "IN", 2) == 0xFFFF)
      install();
   else if (ULCmpB(sd_case.pb, "DI", 2) == 0xFFFF)
      deinstall();
   else
      exit_with_msg(ercSyntax, NLS_CASE_INVALID);
   Exit();

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 Examine the hardware and software environments and determine which
 sequential access service to install.  The choices are:

   NGenSeqService.Run		Convergent or Flemington QIC-02 module
   				 and SCSI
   PertecSeqService.Run		SRP Pertec 1/2" (no QIC, no SCSI)
   SrpSeqService.Run		SRP QIC-02 interface and SCSI

 On the SRP/XE, only, there is a possibility of multiple instances of
 sequential access services.  The Flemington and Convergent QIC-02 modules may
 not both be present on a workstation and the Flemington module is supported
 only if the workstation is executing BTOS. */

private void install(void) {

   unsigned directory_len, erc, filename_len, node_len, password_len,
      runfile_len, volume_len;
   char node[MAX_NODE_LENGTH], volume[MAX_VOLUME_LENGTH],
      directory[MAX_DIRECTORY_LENGTH], filename[MAX_FILENAME_LENGTH],
      password[MAX_PASSWORD_LENGTH], runfile[MAX_FILE_SPECIFICATION_LENGTH];
   sdType *sd_runfile, *sd_volume;
   SysConfigType *sysConfig;

   if (FSrp(NULL)) {
      CheckErc(GetPStructure(ATpConfiguration, 0, &sysConfig));
      if (sysConfig->HardwareType == SP || sysConfig->HardwareType == DP)
         sd_runfile = standard_msg(NLS_PERTEC_SERVICE);
      else
         sd_runfile = standard_msg(NLS_SRP_SERVICE);
   } else
      sd_runfile = standard_msg(NLS_NGEN_SERVICE);
   erc = Chain(sd_runfile->pb, sd_runfile->cb, NULL, 0, 0x80, 0, FALSE);
   if (erc == ercNoSuchFile) {		/* Try finding it at [!Sys] */
      if ((erc = ParseFileSpec(0, sd_runfile->pb, sd_runfile->cb, FALSE, node,
                               &node_len, volume, &volume_len, directory,
                               &directory_len, filename, &filename_len,
                               password, &password_len, TRUE, 0)) != ercOK)
         ErrorExit(ercSyntax);
      sd_volume = standard_msg(NLS_SERVER_SYS_VOLUME);
      memset(runfile, 0, sizeof(runfile));
      CheckErc(BuildFileSpec(0, runfile, &runfile_len, NULL, sizeof(runfile),
                             TRUE, node, node_len, sd_volume->pb,
                             sd_volume->cb, directory, directory_len,
                             filename, filename_len, FALSE, password,
                             password_len, TRUE, 0));
      erc = Chain(runfile, runfile_len, NULL, 0, 0x80, 0, FALSE);
   }
   ErrorExit(erc);		/* Not in the normal course of events */

}

pragma Page(1);
/*-----------------------------------------------------------------------------
 After identifying ourselves, attempt to send a deinstall request to the
 designated Sequential Access Service.  If the request is not served or cannot
 get where it is going because the device name is not in the route table,
 conclude that the service is not installed.  Otherwise, the service will
 deinstall the device specified.  If no other devices are served, the
 partition may be vacated before we exit. */
 
private void deinstall(void) {

   seq_access_deinstall_rq_type deinstall_rq, *response;
   char device[MAX_DEVICE_LENGTH], device_spec[MAX_FILE_SPECIFICATION_LENGTH],
      password[MAX_PASSWORD_LENGTH];
   unsigned device_len, erc, i, node_len, password_len, position_len, ph;
   sdType nls_parms[] = {(void *) device_spec, 0,
                         (void *) &erc, sizeof(erc)};
   sdType sd_device;

   memset(&deinstall_rq, 0, sizeof(deinstall_rq));
   deinstall_rq.s_cnt_info = 6;
   deinstall_rq.n_req_pb_cb = 1;
   deinstall_rq.n_resp_pb_cb = 1;
   CheckErc(QueryDefaultRespExch(&deinstall_rq.exch_resp));
   deinstall_rq.rq_code = SEQ_ACCESS_DEINSTALL_RQ_CODE;
   deinstall_rq.ph = (void *) &ph;
   deinstall_rq.ph_size = sizeof(ph);
   if (CSubparams(DEVICE_NAME_PARAM) == 0) {
      memcpy(&sd_device, standard_msg(NLS_DEFAULT_SEQ_DEVICE),
             sizeof(sd_device));
      CheckErc(RgParamSetSimple(DEVICE_NAME_PARAM, &sd_device));
   }
   for (i = 0; i < CSubparams(DEVICE_NAME_PARAM); i++) {
      CheckErc(RgParam(DEVICE_NAME_PARAM, i, &sd_device));
      if (     ParseFileSpec(0, sd_device.pb, sd_device.cb, FALSE, NULL,
                             &node_len, device, &device_len, NULL, NULL, NULL,
                             &position_len, password, &password_len,
                             FALSE, 3) != ercOK
            || node_len != 0 || position_len != 0)
         ErrorExit(ercInvalidTapeSpecification);
      memset(device_spec, 0, sizeof(device_spec));
      BuildFileSpec(0, device_spec, &nls_parms[0].cb, NULL, last(device_spec),
                    FALSE, NULL, 0, deinstall_rq.device_name = device,
                    deinstall_rq.device_name_len = device_len, NULL, 0, NULL,
                    0, FALSE, NULL, 0, FALSE, 0);
      if ((erc = Request(&deinstall_rq)) == ercOK) {
         do
            erc = Wait(deinstall_rq.exch_resp, &response);
         while (erc == ercOK && response != &deinstall_rq);
         if (erc == ercOK)
            erc = deinstall_rq.erc_ret;
      }
      if (erc == ercOK) {
         log_msg(NLS_DEVICE_DEINSTALLED_OK, nls_parms, sizeof(nls_parms));
         if (ph != 0) {
            if ((erc = VacatePartition(ph)) == ercOK)
               erc = RemovePartition(ph);
            if (erc == ercOK)
               log_msg(NLS_SERVICE_DEINSTALLED_OK, NULL, 0);
            else
               log_msg(NLS_SERVICE_DEINSTALLATION_ERROR, nls_parms,
                       sizeof(nls_parms));
            break;
         }
      } else if (erc == ercServiceNotAvail)
         exit_with_msg(ercOK, NLS_SERVICE_NOT_INSTALLED);
      else if (erc == ercInvalidTapeSpecification || erc == ercNoSuchVolume)
         log_msg(NLS_DEVICE_UNKNOWN, nls_parms, sizeof(nls_parms));
      else
         log_msg(NLS_DEVICE_DEINSTALLATION_ERROR, nls_parms,
                 sizeof(nls_parms));
   }

}
