;  R q   I n t e r f a c e
;
;  The code to which all OS Request procedural interface routines transfer.
;
;  The task is to build an rq on the stack from the parameters
;  pushed on the stack (by the user) and from a table,
;  the address of which is found via the hack stored on the
;  stack (pushed as the CS-IP-ret by an 'INT n' instruction
;  executed by the calling application at an absolute address).
;  See also the assembly module RqLabl.asm for the PUBLIC addresses
;  used by the application.
;
;  The table is peculiar to the request-type (Read is different from
;  Write is different from Open . . .).
;  Functions:
;  1. The table and the arguments are checked for consistency.
;
;  2. Here the request-block is built on the stack under control of
;     the table indicated by the application.
;
;  3. The request is sent, and a reply is awaited.
;
;  4. The erc is loaded in AX, the stack is incremented by the
;     appropriate amount, and a return is executed.
;
; Register Usage:
;
;   AX:	various
;   BX: oTab (ra of the arg/rq translation table),
;	incremented as we process the table byte by byte.
;   CX: rq, cVars, cCons (note that CL also works, since < 128)
;   DL: cbRq (since < 128)
;   DH: cVars
;   SP: usual
;   BP: as in PLM86, at old BP, on top of rq and args
;   SI: oArg (>=BP+8+cbRq)
;   DI: iRq (as in [BP+oRq+DI])
;   ES: various, often RcLookUp tcode table base
;
;
; The stack and registers look like this once the routine
; gets initialized:
;
;		  stack (words)				table (bytes)
;		+---------------+
;	SP->|	FLAG		|
;   BP->|   old BP		|
;BP+pRqRet-> .ra 		|
;		|    .sa 		|
;		|    exchSave	|		+-------+
;BP+rgbRq->The rq under	|		| cbRq	|<--ES:BX
;   :	|  construction	|		| cVars	|		:
;   V	|  (size cbRq)	|		| var0	|		:
;   DI	|				|		| ...	|		:
;		|   BP			|		| varn-1|		:
;		|   IPret		|		| cCons	|		:
;		|   CSret		|		|con0offset		:
; SI--->|   argN-1		|		|con0val|		:
; A		|   ...			|		| ...	|		:
; :		|   ...			|		|conN-1offset	V
; SI--->|   arg0		|		|conN-1val<-----BX
;		|   {userNum}	|		+--------+
;
;	entries in {} are only present when using the Alt interface (FLAG=TRUE).
;
%IF (not(%*Isdef(%CTOSp))) THEN (%Define(CTOSp)(0))FI
%IF(NOT %*IsDef(%ctosv)) THEN (%SET(ctosv,0))FI
$EJECT
;
; Local EQUs:
;
islJumpTable equ 7Ch; ldt selector 7Ch -- index is 15-- for trap jump table
iSgFirst EQU 10h
oFLAG		EQU 0-2
pRqRetRa	EQU 2	; DECLARE pRqRet POINTER
pRqRetSa	EQU 4
exchRespSave	EQU	6
sizeLocals	EQU	6

oAltUserNum EQU 2
bitAltUser	EQU 01h	; set in FLAG when using Alternate proc interface
bitDontWait	EQU 02h	; If not using def resp exch, doesn't Wait()
bitUnLock	EQU 04h	; UnLockInContext after altRequest

rgbRq		EQU sizeLocals+2	; DECLARE rgbRq (1) BYTE
rqErc		EQU rgbRq+8; DECLARE rqErc WORD AT(@rgbRq(8))
rqExchResponse	EQU rgbRq+6
rqUserNo 	EQU rgbRq+4
rqRc		EQU rgbRq+10

ercRqBadTable 	EQU 30
ercNoSuchRc 	EQU 31
ercWrongRqSync	EQU 32
wDummyLSC		EQU 0A3D1h
;
; Arg/Rq transform-table EQUs
;
bOrWd		equ	01h; (msb of byte is flag in constant-table
			   ; but is shifted into lsb of AH)
;
; Pcb EQUs
;
;ppPcbCurrent	EQU 24Ch; globally available pointer to current pcb
exchgSync	EQU 12
userNo		EQU 14

liMsgNoSuchRc EQU 60005	; msg number in plog msg file

$EJECT
; The table RcLookUp is defined by the user in module Request.asm.  The table
; is constructed in segment "OCode" of class "const" and group "DGroup".  It
; contains  offsets to tables constructed in segment
; "TCode" of class "memory".  Each table is defined as
; described in the comments at the beginning of this file.
;
; With the addition of request levels,
;  RcLookUp becomes many arrays, one for each level.
;  Each array may have a different max request code, and a different base for
;  its tables.
; This gives rise to new arrays:
;  rgPrcLookUp(even levels) rgRcMax(level) rgPrcLookUpBase(even levels)
;


EXTRN Crash:far
EXTRN FEqP:far
EXTRN MediateTrap:far
EXTRN LogMisc:FAR
EXTRN GRequest:NEAR, GWait:NEAR, KernelKillUser:NEAR

%if (%ctosp and not %ctosv) then(
EXTRN LockInContext:far, UnLockInContext:far
)fi

%if(%ctosp)then(
extrn AliasRealRequestBlock:far, DealiasRealRequestBlock:far
)fi

; Mention  'const' segments to keep them in together - order does not matter

RqGroup group rqSeg0, rqSeg1, rqSeg2, rqSeg3, OCode
DGroup  group data, rqSeg5, rqSeg6, rqSeg7, rqSeg8, rqSeg9

data segment public 'data'
extrn oPcbRun:word
%if(%ctosp)then(
extrn sgRealInterface:word
public sgRqInterface
sgRqInterface dw seg RqInterface
)fi
data ends

rqSeg0 SEGMENT PUBLIC 'oemseg'
rqSeg0 ENDS

rqSeg1 SEGMENT PUBLIC 'oemseg'
rqSeg1 ENDS

rqSeg2 SEGMENT PUBLIC 'oemseg'
rqSeg2 ENDS

rqSeg3 SEGMENT PUBLIC 'oemseg'
rqSeg3 ENDS


rqSeg5 SEGMENT PUBLIC 'const'
rqSeg5 ENDS

rqSeg6 SEGMENT PUBLIC 'const'
EXTRN rgPrcLookUp:DWORD
rqSeg6 ENDS

rqSeg7 SEGMENT PUBLIC 'const'
EXTRN rgPrcLookUpBase:DWORD
rgPrcLUBaseWord EQU WORD PTR rgPrcLookUpBase
rqSeg7 ENDS

rqSeg8 SEGMENT PUBLIC 'const'
EXTRN rgRcMax:WORD
rqSeg8 ENDS

rqSeg9 SEGMENT PUBLIC 'const'
EXTRN rgPrgLocalServiceCode:DWORD
rqSeg9 ENDS

OCode SEGMENT PUBLIC 'oemseg'
OCode ENDS

TCode SEGMENT PUBLIC 'memory'
TCode ENDS


Kernel SEGMENT PUBLIC 'code'

KGroup GROUP Kernel
Assume CS:Kernel

$EJECT
; CS:IP contain the request code, encoded in their bits.
; Old real-mode format, level 0 and 16:
; CS | NEG rq15..rq3 |000|		IP |rq11..rq3|000|rq2rq1rq0|0| + 210h
;
; New real-mode format:
; CS |0000*|rq11..rq3|rq15..rq13|	IP |   NEG CS    |rq2rq1rq0|0| + 210h
;
; *1111 if CS>21, so IP doesn't try to go negative: ^			  ^
; Bit rq12 is not needed, because odd rq levels don't have proc. interface.
;
; Protected-mode format, levels 0 to 15:
; CS |0000 000|rq15rq14rq13rq11rq10rq9|000| + 80h	IP |0000 000rq8..rq0|

; An Alternate interface lets the caller give another argument like this -
;	erc=AltOpen(userNum, ...);
;  where the new argument comes first, and is followed by the regular args.
; When userNum = 0, the caller's userNum is used.
; Alt interface is signaled by rq code having 800h added to it. This also
;  halves the useful size of each level, from 4096 to 2048, which is no loss.
; Alt int for Usr rqs takes 32 away from level 0 because 0FFFFh+800h=7FFh,
;  which looks like an rq in level 0.

PUBLIC	RqInterface
RqInterface PROC FAR

; Don't disturb any registers if the INT is from a trap table
	push BP
	mov  BP,SP
	cmp  WORD PTR [BP+4],islJumpTable
	pop  BP
	jne  NotaTrap
	jmp  MediateTrap
NotaTrap:
	XOR  DX,DX		; regular RqLabl, FLAG=FALSE


	pop  AX			; AX = (IPret MOD 16)/2	;/*3 bits of rc*/
	sub  AX, 2		; IP pointed to the byte after INT6
	pop  CX			; CS
	popf			; Restore the flag
	mov  SI,SP		; oArgs = SP
	
	push bp         ; for debugger stack trace
	mov  bp, sp
	
	push ds
	mov di, DGroup
	mov ds,di
assume ds:DGroup

;	Decode CX,AX by one of three schemes.

%if(%ctosp)then(
	mov  di, sgRealInterface
	cmp  [bp+4], di
	je RealModeRqInterface

;on entry CX = CS, AX = IP
;on exit, CX = rqCode, DI = level*2

; riSg = SHR(FixupCs, 3) - iSgFirst
	shr cx, 3
	sub cx, iSgFirst

; level (* 2) = SHR(riSg, 3) * 2 (* 2)
	mov di, cx
	shr di, 1
	and di, 0FFFCh

; rqCode = FixupIp + (512 * (riSg AND 7))
	mov ch, cl		; riSg*256
	shl ch, 1		;         *2 -> *512
	and cx, 0E00h	; ch = 0000 rq11rq10rq9 0
	add cx, ax

	jmp TryRq

RealModeRqInterface:
)fi

	test CX,7		; Old request code format has low 3 bits = 0
	jz  OldRq

NewRq:
	mov	DI,CX		; save CS, to extract level bits later
	and	CX,0FF8h	; (CS AND 0FF8h) is rq 11-3
	and	AX,0Fh		; (IP AND 0Fh) is (rq 2-0)*2
	shr	AX,1		; rq 2-0
	or	CX,AX		; (CS AND 0FF8h) OR (IP AND 0Fh)
					;  is rq 11-0 = rcode

	and	DI,7		; (CS AND 7) is rq 15-13 = level/2
	shl	DI,2		; level (only even levels have procedural interface)
	Jmp TryRq

OldRq:
	and  AX,000Fh
	shr  AX,1
	neg  CX			; saveRc = -CSret
	add  CX,AX		;     + AX
; sign bit set by this add used in jns below.

; Four possible request cases:
;	level 0			0001h to 0000h+rgRcMax[0]
;	level 0 Alt		0801h to 0800h+rgRcMax[0]
;	level 16		FFE0h to FFFFh
;	level 16 Alt	07E0h to 07FFh
	mov  DI, 0		; assume level 0
	Jns  TryRq
; Sign bit set can only be "Usr" requests, set to level 16.
	mov  DI, 10h*2
	not  CX

TryRq:
; assert - CX is rqCode, DI is level*2

; Three possible request cases in CX:
;1	level n			001h to 000h+rgRcMax[n]
;2	level n Alt		801h to 800h+rgRcMax[n]
;3	level 16 Alt	7E0h to 7FFh (DI is 0)
	cmp	CX,rgRcMax[DI]	; rcode too large?
	jb	CheckForDummyLSC

; rcode was too large, maybe because it was AltRqInterface
;	Preserve DI,CX,SI,BP
	xor DL,bitAltUser	; Alt RqLabl, FLAG=TRUE
;	Try again with rq (CX) -800h, Alt flag set.
	sub CH,8		; Alt rq interface looks like rq+800h in a level
	jns RetryAlt	; CX case 2!  It was AltRqInterface.
; CX case 1 or 3, distinguish by level in DI
	cmp DI, 0
	jne ExitNoSuchRequest	; CX case 1.
; Almost certainly CX case 3.
; Note if CX case 1,level was 0, rq out of range then RetryAlt check will catch.
	mov DI, 10h*2
	not CX

RetryAlt:
	cmp	CX,rgRcMax[DI]	; rcode too large?
	jb	CheckForDummyLSC
	
ExitNoSuchRequest:
; log the error and the request code
	CMP  DI, 10h*2
	JNE  @0
	NOT  CX
@0:
; rebuild request code for log msg
; DI - level*2 or 20h (for rc -1..-32)
	SHL  DI, 11
	OR   CX, DI
	AND  CX, 0f7ffh	; clear altRq bit
	PUSH CX		; rqCode
	MOV  SI, SP
	PUSH liMsgNoSuchRc
	PUSH SS
	PUSH SI
	PUSH 2
	CALL LogMisc ; 	CALL LogMisc(iMsg, @rqCode, 2)
	POP  AX

	mov	AX,ercNoSuchRc	; CALL ErrorExit ercNoSuchRc
	JMP	KernelKillUser

CheckForDummyLSC:
; DI = level*2 also indexes PTR array with elements only for even levels.
; CX = level relative request code

; If local service code indicates a dummy request, kill user because
; the number of parameters on the stack is unknown, hence, return from
; rqInterface will be incorrect.

	shl di, 1 ; rgPrgLocalService code is a table containing all levels
	les	BX,rgPrgLocalServiceCode[DI]
	shr di, 1 
	add	BX,CX
	add	BX,CX
	cmp word ptr es:[bx], wDummyLSC
	je ExitNoSuchRequest

	les	BX,rgPrcLookUp[DI] ; ES:BX->rcLookUpN
	add	BX,CX		; offset rcode into rcLookUpN array of words
	add	BX,CX
	mov	BX,ES:[BX]	; BX=rcLookUpN[rcode]=offset from some base

	mov	AX,rgPrcLUBaseWord+2[DI] ; sa
	add	BX,rgPrcLUBaseWord[DI]	 ; ra+offset=start of table
	mov es, ax
assume es:nothing

	POP DS	; replace user DS, so its there in the debugger
ASSUME DS:nothing

; ES:BX=oTab^
; CX=rcode
; make whole rq by appending level to rcode
	mov	AX,DI	;level*2
	shl	AL,3	;	*16
; level 16?(Usr) Then c=1, AX=0
	jnc	@1
	not	CX	;subsequent OR will do nothing if Usr
@1:
	or	CH,AL	;append level nybble to rcode
; CX=rq
RcOK3:	mov	AL,ES:[BX]	; cbRq = oTab^	; /*1st byte of table is cbRq*/
	xor  AH,AH
	sub  SP,AX		; /*make room for rq
	sub  SP,sizeLocals	;   and for pRqRet, exchRespSave
	push BP			;   save old BP on stack */
	mov  BP,SP

	push DX			; flag - TRUE=AltRqLabl  FALSE=regular rqlabl
	xchg AX,DX		; DL=cbRq DH=0

	inc  BX			; oTab = oTab + 1
	mov  DH,ES:[BX]		; cVarsSave = oTab^

	add  SI,2		; oArgs = oldSP +(cVarsSave*2)+ 2
	mov  AL,DH
	xor  AH,AH
	shl  AX,1
	add  SI,AX		; /*oArgs=offset of bottom-most arg (arg0)*/

				; /*build the rq*/
	mov  DI,BP		; rgbRq = 0
	add  DI,rgbRq
	mov  AX,SS
	push CX			; stack rc
	push ES			; stack saTab
	push BX			; stack oTab
	mov  ES,AX		; now ES:DI is @rq
	cld			; DF=0 for increment
	mov  CL,DL		; CX=cbRq
	xor  CH,CH
	xor  AX,AX		; AX=0
	rep stos BYTE PTR ES:[DI]

	test BYTE PTR[BP+oFLAG],bitAltUser
	jz   NotAlt

	mov  AX, ES:[SI+oAltUserNum]
	mov  [BP+rqUserNo],AX

NotAlt:
	mov  ax, DGroup
	mov  es, ax
assume es:DGroup
	mov  bx, oPcbRun
	mov  AX,ES:[BX+exchgSync]
	mov  [BP+rqExchResponse],AX
	mov  [BP+exchRespSave],AX
DefUser:

	pop  BX			; restore oTab
	pop  ES			; restore saTab
assume es:nothing
	pop  AX			; rq.Rc = saved rc
	mov  [BP+rqRc],AX
	mov  CL,DH		; cVars = cVarsSave
		;CH = 0 (above)

	inc  BX			; oTab = oTab + 1
DoVars: jcxz DoVarsEnd		; DO WHILE cVars > 0
	mov  AL,ES:[BX]		;   var = oTab^
	shl  AX,1		;   /*AH lsb is now msb of var*/
	shr  AL,1		;   /*AL=l.s. 7 bits, or offset*/
	cmp  AL,DL		;   IF var.offset < cbRq THEN
	jb   RqOkTable		;     GO TO RqOkTable
RqBadTable:			; The table is inconsistent.
	mov  AX,ercRqBadTable
	push AX
	call Crash

RqOkTable:
	test AH,bOrWd		;   IF var.fVarIsWord THEN
	jz   VarIsByte
	xor  AH,AH		;     word@(rgbRq(var.offset))
	mov  DI,AX
	mov  AX,SS:[SI]		;      = word@oArgs
	mov  SS:[BP+rgbRq+DI],AX
	jmp  NextVar
VarIsByte:			;   ELSE
	xor  AH,AH		;     byte@(rgbRq(var.offset))
	mov  DI,AX
	mov  AL,SS:[SI]		;     = byte@oArgs
	mov  SS:[BP+rgbRq+DI],AL
NextVar: inc  BX		;   oTab = oTab + 1
	dec  SI			;   oArgs = oArgs - 2
	dec  SI
	loop DoVars		;   cVars = cVars - 1
DoVarsEnd:			;   END

	;  R e a d   A l l   C o n s t a n t s

	mov  CL,ES:[BX]		; cCons = oTab^
;	xor  CH,CH done by loop

	inc  BX			; oTab = oTab + 1
DoCons: jcxz DoConsEnd		; DO WHILE cCons > 0
	mov  AL,ES:[BX]		;   offset = oTab^
	cmp  AL,DL		;   IF offset >= cbRq THEN
	jae  RqBadTable		;     GO TO RqBadTable
	inc  BX			;   oTab = oTab + 1

	xor  AH,AH		;     byte@(rgbRq(offset))
	mov  DI,AX
	mov  AL,ES:[BX]		;     = oTab^
	mov  SS:[BP+rgbRq+DI],AL
	inc  BX			;   oTab = oTab + 1
	loop DoCons		;   cCons = cCons - 1
DoConsEnd:			;   END

	push DX			; save DX
; push pRq for Request
	push SS	
	lea  bx,[BP+rgbRq]
	push bx
%if(%ctosp)then(
; if caller is real mode, alias the request block
; determine mode of caller by return address
	mov	di, bp
	mov di, word ptr ss:[di] ; go back one frame
	mov di, ss:[di+4] ; csRet
	push ds
	mov ax, DGroup
	mov ds, ax
assume ds:dGroup
	cmp di, sgRealInterface
	jne DoRequest
	push ss
	pop es
; es:bx is pRq
	call AliasRealRequestBlock
DoRequest:
	pop ds
assume ds:nothing

%if(not %ctosv)then(
;If alt rq, ensure caller unswappable during wait
	test BYTE PTR [BP+oFlag], bitAltUser
	jz   CallRequest
	test BYTE PTR [BP+oFlag], bitDontWait
	jnz  CallRequest

	push 0			; lock pcbRun
	call LockInContext
	or   AX, AX
	jnz  CallRequest
	or   byte ptr [bp+oFlag], bitUnLock
)fi

CallRequest:
)fi

	push cs
	call GRequest	; near call, rqInterface and kernel share the segment
	and  ax,ax		; IF erc <> 0 THEN
	jnz  RqExit		;   Go To RqExit

; IF fDontWait THEN don't wait (not using default resp exch)
	test BYTE PTR [BP+oFLAG],bitDontWait
	jnz  RqExit		;   Go To RqExit

				; erc = Wait(exchRespSave, @pRqRet)
	push [BP+exchRespSave]
	push SS		; sa(@pRqRet)
	lea  AX,[BP+pRqRetRa]
	push AX	; ra(@pRqRet)
	push cs
	CALL GWait		; near call, rqInterface and kernel share the segment
	and  ax,ax		; IF erc <> 0 THEN
	jnz  RqExit		;   Go To RqExit
	;			; IF pRqRet <> @rq THEN
; if raRq = raMsgret then forego linear comparision (FEqP)
	LEA  AX, [BP+rgbRq]
	CMP  AX, [BP+pRqRetRa] 
	JE   PtrsOk
	push SS
	lea  AX,[BP+rgbRq]
	push AX
	push [BP+pRqRetSa]
	push [BP+pRqRetRa]
	call FEqP
	rcr al, 1
	jc PtrsOk

PUBLIC WrongRq
WrongRq:			;    CALL ErrorExit(ercWrongRqSync)
	mov  AX,ercWrongRqSync
	jmp  KernelKillUser	; Never returns

PtrsOk:

%if(%ctosp)then(
; If caller is real mode, dealias the request block so that
; alias selectors will be deallocated.  Determine mode of caller
; by return address.
	mov	di, bp
	mov di, word ptr ss:[di] ; go back one frame
	mov di, ss:[di+4] ; csRet
	push ds
	mov ax, DGroup
	mov ds, ax
assume ds:dGroup
	cmp di, sgRealInterface
	jne StoreBackErc
	les bx, dword ptr [BP+pRqRetRa]
; es:bx is pRq
	call DeAliasRealRequestBlock
StoreBackErc:
	pop ds
assume ds:nothing
)fi
	mov  AX,[BP+rqErc]	; ercRet = rq.erc

RqExit:				; /*AX contains erc*/

%if (%ctosp and not %ctosv) then(
;If we disabled swap for alt call wait, restore its prior state now
	test BYTE PTR [BP+oFlag], bitUnLock
	jz   FinishRqExit

	push ax			;save erc
	push 0
	call UnLockInContext
	pop  ax

FinishRqExit:
)fi
	pop  DX			; Restore DX
	pop  CX			; FLAG
	pop  BP			; restore BP
	mov  BL,DL		; SP = SP + cbRq + SIZE(ppRqRet) + SIZE(exchRespSave)
	xor  BH,BH
	add  BX,sizeLocals
	add  SP,BX		; /*this puts stack back to level at call
					; now IPret and CSret are on top of stack*/
	mov  BL,DH		; BX = cbArgs	;/*for ret-N simulation below*/
	xor  BH,BH
	jcxz PopArgs
	inc  BX			; AltRqLabl, has extra argument word (userNum)
PopArgs:
	shl  BX,1

	pop bp      ; restore BP for debugger stack trace

	pop  DI			; store CS & IP . . .
	pop  SI
	add  SP,BX		; cut back the stack by cbArgs (a la ret-N)
	push SI			; put CS & IP back
	push DI
	ret

RqInterface ENDP

Kernel 	ENDS

END

; Change Log
;
; 3/14/86  DR - removed RqGroup, reordered const segments
; 2/17/87  DR - call dealiasing procedure to deallocate RMOS selectors
; 6/23/87  JM - TCode is class 'memory' so it can be moved and released.
; 9/16/87  JM - modify VM source for real mode (9.9).
; 4/25/88 MTR - ensure swap disabled during alt call wait
; 10/25/88 MTR - use [Un]LockInContext during alt call
; 03/21/89 JAY - isolate pointer comparison for rm/pm.
; 01/17/90 JA - Fix entire rqCode cs/ip hack decoding.  RMOS had Alt bug.
; 05/16/90 DR - KillUser if local service code indicates dummy request.
; 10/23/90 JM - merge: 9/26/89  DR - added sgRqInterface.
; 12/04/90 JF - Jump to MediateTrap if INT CD from trap jump table
; 08/09/91 JA - islJumpTable is 7Ch.
; 09/25/91 JM - log erc 31 with request code.
; 02/08/92 JM - combine with kernel segment, and misc optimization.
; 05/14/92 JM - reset altRq bit when logging erc 31.
; 11/16/92 JF - Remove unused reference to ContextStatus
