{$LINESIZE:96}
{$TITLE:'04 FEB 85: Interrupt Service Routines for General Synchronous Communications'}
{$PAGESIZE:60}

{$DEBUG-}
{$OCODE+}

module GenSyncISR [];

{$INCLUDE:'<PascalEdf>CTOS.Edf'}
{$INCLUDE:'<PascalEdf>ProcessMgt.Edf'}
{$INCLUDE:'<PascalEdf>TimerMgt.Edf'}
{$INCLUDE:'<PascalEdf>InterruptHandlers.Edf'}
{$INCLUDE:'<PascalEdf>iApx186.Edf'}
{$INCLUDE:'<PascalEdf>8274MPSC.Edf'}
{$PAGE+}
{$INCLUDE:'<PascalEdf>GenSync.Edf'}
{$PAGE+}
var [public]
   ccb :array[0..1] of ccbType;

value
   ccb[0].iLine := 0;				{Constant info...}
   ccb[1].iLine := 1;
   ccb[0].open := false;			{Initial info...}
   ccb[0].txOn := false;
   ccb[0].txIob.trb.counter := 0;
   ccb[0].txIob.trb.counterReload := 0;
   ccb[0].txIob.trb.cEvents := 0;
   ccb[0].txIob.trb.ercRet := ercOK;
   ccb[0].txIob.txState := txIdle;
   ccb[0].rxOn := false;
   ccb[0].rxIob.trb.counter := 0;
   ccb[0].rxIob.trb.counterReload := 0;
   ccb[0].rxIob.trb.cEvents := 0;
   ccb[0].rxIob.trb.ercRet := ercOK;
   ccb[0].rxIob.rxState := rxIdle;
   ccb[1].open := false;
   ccb[1].txOn := false;
   ccb[1].txIob.trb.counter := 0;
   ccb[1].txIob.trb.counterReload := 0;
   ccb[1].txIob.trb.cEvents := 0;
   ccb[1].txIob.trb.ercRet := ercOK;
   ccb[1].txIob.txState := txIdle;
   ccb[1].rxOn := false;
   ccb[1].rxIob.trb.counter := 0;
   ccb[1].rxIob.trb.counterReload := 0;
   ccb[1].rxIob.trb.cEvents := 0;
   ccb[1].rxIob.trb.ercRet := ercOK;
   ccb[1].rxIob.rxState := rxIdle;
{$PAGE+}
{Public procedures to mask and unmask interrupts from the RS-232C
 communications channels.  Once masked, the routines that operate
 at the interrupt levels need not worry about the timer interrupts, since
 these have a lower priority.  Other routines must make their own arrangements
 with respect to the timer interrupts.  Note:  These are for the N'Gen only.}

procedure MaskRS232CInterrupts [public];
begin
   Disable;	{Critical section while 80186 interrupt mask is referenced}
   Outword(mask, Inword(mask) or int1Mask);
   Enable;
end;

procedure UnmaskRS232CInterrupts [public];
begin
   Disable;	{As above...}
   Outword(mask, Inword(mask) and (not int1Mask));
   Enable;
end;

{Public procedures to shut down transmission and/or reception over the com-
 munications channel and set the state variables to reflect this condition.
 These procedures are used both by the interrupt service routines and by the
 termination request process running at a lower priority.  Because of this,
 it is taken for granted by these procedures that interrupts are masked
 when they are called; it could be unhealthy to do otherwise.}

procedure TerminateTx(var ccb :ccbType) [public];
begin
   LockOut(ccb.ctrl, errorReset or wr5);
   LockOut(ccb.ctrl, ccb.wr5Init);
   ccb.txIob.trb.counter := 0;
   ccb.txIob.txState := txIdle;
   ccb.txOn := false;
end;

procedure TerminateRx(var ccb :ccbType) [public];
begin
   LockOut(ccb.ctrl, errorReset or wr3);
   LockOut(ccb.ctrl, ccb.wr3Init);
   ccb.rxIob.trb.counter := 0;
   ccb.rxIob.rxState := rxIdle;
   ccb.rxOn := false;
end;
{$PAGE+}
{Common procedures to terminate transmission, reception or both when an error
 is discovered by the communications interrupt service routines.  These make
 use of the public procedures above and, after the channel has been quiesced,
 then post an error return code to the completion service which then notifies
 the user.}

procedure abortTx(var ccb :ccbType; erc :ercType);
begin
   TerminateTx(ccb);
   if (ccb.wr5Init and dtr) = 0 then
      ccb.rs232C := ccb.rs232C - [dataTerminalReady];
   if (ccb.wr5Init and rts) = 0 then
      ccb.rs232C := ccb.rs232C - [requestToSend];
   ccb.txIob.trb.ercRet := erc;
   erc := PSend(ccb.txIob.complExch, ads ccb.txIob);
end;

procedure abortRx(var ccb :ccbType; erc :ercType);
begin
   TerminateRx(ccb);
   ccb.rxIob.trb.ercRet := erc;
   erc := PSend(ccb.rxIob.complExch, ads ccb.rxIob);
end;

procedure abortTxRx(var ccb :ccbType; erc :ercType);
begin
   abortTx(ccb, erc);
   abortRx(ccb, erc);
end;
{$PAGE+}
{Public procedure to maintain the one-word data structure in the Ccb that
 reflects the status of the RS232C modem control and status leads.  These are
 retrieved from both Read Register 0 of the communications chip and from the
 Extended Control and Status Register.  Note that the ECSR has information for
 both channels and the information is aligned as appropriate for the channel to
 be examined.}

function SetRs232C(rr0Image, ecsrImage :byte; iLine :sint) :rs232CType
          [public];
begin
   SetRs232C := [];
   if (rr0Image and cts) = cts then
      SetRs232C := result(SetRs232C) + [clearToSend];
   if (rr0Image and rlsd) = rlsd then
      SetRs232C := result(SetRs232C) + [carrierDetect];
   if iLine = 0 then			{ECSR has status for both channels}
      ecsrImage := ecsrImage div 16;
   if (ecsrImage and secRxData) = 0 then
      SetRs232C := result(SetRs232C) + [secondaryRxData];
   if (ecsrImage and dsr) = 0 then	{NOTE: Read bits in ECSR inverted...}
      SetRs232C := result(SetRs232C) + [dataSetReady];
   if (ecsrImage and ri) = 0 then
      SetRs232C := result(SetRs232C) + [ringIndicator];
   if (ecsrImage and secTxData) = secTxData then	{....write bits not!}
      SetRs232C := result(SetRs232C) + [secondaryTxData];
end;
{$PAGE+}
{Transmit interrupt service routine is state driven: the next character
 presented to the communications channel is a product of the current state,
 the communications mode (transparent or not) and the last character written.
 Transmission continues until the character count is satisfied or an error
 condition intervenes.  Note the interaction between this service routine and
 the external interrupt service routine; they are both necessary for messages
 to be transmitted.}

procedure TxISR(iLine :sint) [public];
var
   char :byte;
   ctrlPort :word;
   erc :ercType;
   pCcb :pCcbType;

begin
   pCcb := adr ccb[iLine];		{Establish comm control block}
   ctrlPort := pCcb^.ctrl;		{Optimization for LockOut's}
   if not pCcb^.txOn then [
      LockOut(ctrlPort, resetTxInt or wr5);
      LockOut(ctrlPort, pCcb^.wr5Init);
   ] else with pCcb^.txIob do
      case txState of
        txSyn:	[
                   if pCcb^.nSynChar = 1 then		{Monosync}
                      LockOut(pCcb^.data, pCcb^.synChar[1])
                   else
                      if odd(txSynCount) then	{Bisync odd...}
                	 LockOut(pCcb^.data, pCcb^.synChar[2])
                      else				{Bisync even...}
                	 LockOut(pCcb^.data, pCcb^.synChar[1]);
                   txSynCount := txSynCount - 1;
                   if txSynCount = 0 then
                      if pCcb^.crcEnable then
                	 txState := txSom
                      else
                	 txState := txData;
                ];

        txSom:	[
                   char := pBuffer^;		{Optimize dereferences}
                   if not pCcb^.crcEnable then
                      txState := txData		{Simple...no CRC!}
                   else if txMode = nontransparent then
                      case count of	{Nontransparent section}
                	 0: if (char <> stx) and (char <> soh) then
                	       txState := txData;

                	 1: [
                	       txCrcOn := true;
                	       txState := txData;
                	       LockOut(ctrlPort, resetTxCrc or wr5);
                	       LockOut(ctrlPort, pCcb^.wr5init or (txEnable
                                       or rts or txCrcEnable));
                	    ];
                      end
                   else [			{This section for transparent}
                      txLastChar := char;
                      case count of
                	 0: if char <> dle then
                	       txState := txData;

                	 1: if (char <> stx) and (char <> soh) then
                	       txState := txData;

                	 2: [
                	       txCrcOn := true;
                	       txState := txData;
                	       LockOut(ctrlPort, resetTxCrc or wr5);
                	       LockOut(ctrlPort, pCcb^.wr5init or (txEnable
                                       or rts or txCrcEnable));
                	    ];
                      end;
                   ];
                   LockOut(pCcb^.data, char);
                   pBuffer.r := pBuffer.r + 1;
                   sBuffer := sBuffer - 1;
                   count := count + 1;
                   if sBuffer = 0 then
                      if txCrcOn then
                	 txState := txCrc
                      else
                	 txState := txPad;
                ];

        txData:	[
                   char := pBuffer^;
                   if (txMode = transparent) and txCrcOn then [
                      if txLastChar = dle then [
                	 LockOut(ctrlPort, wr5);
                	 LockOut(ctrlPort, pCcb^.wr5init or (txEnable or rts
                                 or txCrcEnable));
                      ] else if char = dle then [
                	 LockOut(ctrlPort, wr5);
                	 LockOut(ctrlPort, pCcb^.wr5Init or (txEnable or rts));
                      ];
                      if txLastChar = dle then
                	 txLastChar := pad	{Anything but a dle!}
                      else
                	 txLastChar := char;
                   ];
                   LockOut(pCcb^.data, char);
                   pBuffer.r := pBuffer.r + 1;
                   sBuffer := sBuffer - 1;
                   count := count + 1;
                   if sBuffer = 0 then
                      if txCrcOn then
                	 txState := txCrc
                      else
                	 txState := txPad;
                ];

        txCrc:	[
                   LockOut(ctrlPort, resetTxUnder or resetTxInt);
                   txState := txPad;
                ];

        txPad:	[
                   LockOut(pCcb^.data, pad);
                   txPadCount := txPadCount - 1;
                   if txPadCount = 0 then
                      txState := txFin;
                ];

        txFin:	[
                   LockOut(ctrlPort, resetTxInt or wr5);
                   LockOut(ctrlPort, pCcb^.wr5Init);
                   trb.counter := 0;
                   txState := txIdle;
                   pCcb^.txOn := false;
                   if (pCcb^.wr5Init and dtr) = 0 then
                      pCcb^.rs232C := pCcb^.rs232C - [dataTerminalReady];
                   if (pCcb^.wr5Init and rts) = 0 then
                      pCcb^.rs232C := pCcb^.rs232C - [requestToSend];
                   trb.ercRet := ercOK;
                   erc := PSend(complExch, ads (pCcb^.txIob));
                ];

        otherwise
      end;
end;
{$PAGE+}
{External interrupt service routine plays a part in both the transmit and
 receive functions.  Principally, it is entered when CTS goes high and its
 function is to write a SYN character to the channel (the transmit interrupt
 service routine will write the rest of the SYN characters) or it is entered
 when synchronization is detected during the hunt phase and its function is
 to change the receive state to look for the start of message.  Error
 conditions, such as absent DSR for either state, absent CTS for transmit or
 absent RLSD for receive are also detected here and the request in progress
 terminated.}

procedure ExtISR(iLine :sint) [public];
var
   rr0Image, ecsrImage :byte;		{Save data read from 8274}
   ctrlPort :word;			{Reduce pCcb^.ctrl dereferences}
   pCcb :pCcbType;

begin
   pCcb := adr ccb[iLine];		{Establish comm control block}
   with pCcb^ do [
      ctrlPort := ctrl;
      LockOut(ctrlPort, resetExtInt);
      rr0Image := LockIn(ctrlPort);	{Read relevant status from rr0 and...}
      ecsrImage := Input(ecsr);		{...and external status register}
      rs232C := rs232C * [dataTerminalReady, requestToSend];
      rs232C := rs232C + SetRs232C(rr0Image, ecsrImage, iLine);
      if (rxOn or txOn) and not (dataSetReady in rs232C) then
         abortTxRx(pCcb^, ercLostDsr);
      if txOn and (txIob.txState <> txRts) and
            not (clearToSend in rs232C) then
         abortTx(pCcb^, ercLostCts);
      if rxOn and (rxIob.rxState <> rxHunt) and
            not (carrierDetect in rs232C) then
         abortRx(pCcb^, ercLostRlsd);
      if txOn then with txIob do
         case txState of
          txRts:  if clearToSend in rs232C then [
                     txState := txSyn;		{OK, have CTS, start on Syn's}
                     LockOut(ctrlPort, wr5);
                     LockOut(ctrlPort, wr5Init or (txEnable or rts));
                     if nSynChar = 1 then			{Monosync}
                	LockOut(data, synChar[1])
                     else
                	if odd(txSynCount) then		{Bisync odd...}
                	   LockOut(data, synChar[2])
                	else				{Bisync even...}
                	   LockOut(data, synChar[1]);
                     txSynCount := txSynCount - 1;
                     if txSynCount = 0 then
                	if crcEnable then
                	   txState := txSom
                	else
                	   txState := txData;
                  ];

          txSom,
          txData: abortTx(pCcb^, ercTxUnderrun);

          otherwise
         end
      else if rxOn then with rxIob do
         case rxState of
          rxHunt: if (rr0Image and syncHunt) = 0 then
                     rxState := rxSom;

          rxSom,
          rxData,
          rxCrc:  if (rr0Image and syncHunt) = syncHunt then
                     abortRx(pCcb^, ercLostSynchronization);

          otherwise
         end;
   ];
end;
{$PAGE+}
{Receive interrupt service routine is state driven: what is done with the next
 character retrieved from the communications channel is a product of the
 current state, the communications mode (transparent or not) and the last
 character received. Reception continues until a block terminating sequence
 is detected, the character count is overrun or another error condition
 intervenes.  Note the interaction between this service routine and
 the external interrupt service routine; they are both necessary for messages
 to be received.}

procedure RxISR(iLine :sint) [public];
var [extern]
   bitMask :array[0..15] of bitMap;

var
   eom :Boolean;
   char :byte;
   rr0Image, rr1Image, ecsrImage :byte;	{Save data read from 8274}
   ctrlPort :word;			{Reduce pCcb^.ctrl dereferences}
   erc :ercType;
   pCcb :pCcbType;

begin
   pCcb := adr ccb[iLine];		{Establish comm control block}
   ctrlPort := pCcb^.ctrl;
   if not pCcb^.rxOn then [
      LockOut(ctrlPort, wr3);
      LockOut(ctrlPort, pCcb^.wr3Init);
   ] else with pCcb^.rxIob do [
      char := LockIn(pCcb^.data);	{Optimization}
      case rxState of
        rxSom:	[
                   if (count = 0) and (not pCcb^.synStrip) then [
                      LockOut(ctrlPort, wr3);	{Clear SYN load inhibit}
                      Lockout(ctrlPort, pCcb^.wr3Init or rxEnable);
                   ];
                   if not pCcb^.crcEnable then
                      rxState := rxData		{Simple...no CRC!}
                   else if rxMode = nontransparent then
                      case count of	{Nontransparent section}
                	 0: if (char <> stx) and (char <> soh) then
                	       rxState := rxData;

                	 1: [
                	       rxCrcOn := true;
                	       rxState := rxData;
                	       LockOut(ctrlPort, resetRxCrc or wr3);
                	       LockOut(ctrlPort, pCcb^.wr3init or (rxCrcEnable
                                       or rxEnable));
                	    ];
                      end
                   else [			{This section for transparent}
                      rxLastChar := char;
                      case count of
                	 0: if char <> dle then
                	       rxState := rxData;

                	 1: if (char <> stx) and (char <> soh) then
                	       rxState := rxData;

                	 2: [
                	       rxCrcOn := true;
                	       rxState := rxData;
                	       LockOut(ctrlPort, resetRxCrc or wr3);
                	       LockOut(ctrlPort, pCcb^.wr3init or (rxCrcEnable
                                       or rxEnable));
                	    ];
                      end;
                   ];
                   pBuffer^ := char;
                   pBuffer.r := pBuffer.r + 1;
                   sBuffer := sBuffer - 1;
                   count := count + 1;
                ];

        rxData:	[
                   if (rxMode = transparent) and rxCrcOn then [
                      if rxLastChar = dle then [
                	 LockOut(ctrlPort, wr3);
                	 LockOut(ctrlPort, pCcb^.wr3init or (rxEnable
                                 or rxCrcEnable));
                      ] else if char = dle then [
                	 LockOut(ctrlPort, wr3);
                	 LockOut(ctrlPort, pCcb^.wr3Init or rxEnable);
                      ];
                   ];
                   pBuffer^ := char;
                   pBuffer.r := pBuffer.r + 1;
                   sBuffer := sBuffer - 1;
                   count := count + 1;
                   if rxLastChar = dle then
                      if rxMode = nontransparent then
                	 eom := (pCcb^.dleTermTable[ord(char div 16)] *
                                 bitMask[ord(char mod 16)]) <> []
                      else
                	 eom := (pCcb^.termTable[ord(char div 16)] *
                                 bitMask[ord(char mod 16)]) <> []
                   else if rxMode = nontransparent then
                      eom := (pCcb^.termTable[ord(char div 16)] *
                	      bitMask[ord(char mod 16)]) <> []
                   else
                      eom := false;
                   if eom then [
                      rxState := rxCrc;
                      LockOut(ctrlPort, wr3);		{SYN in BCC is OK!}
                      LockOut(ctrlPort, pCcb^.wr3Init or (rxCrcEnable or
                	      rxEnable));
                   ] else if (sBuffer = 0) then
                      abortRx(pCcb^, ercBufferOverflow)
                   else if rxLastChar = dle then
                      rxLastChar := pad			{Anything but a dle!}
                   else
                      rxLastChar := char;
                ];

        rxCrc:	[
                   rxCrcDelay := rxCrcDelay - 1;
                   if rxCrcDelay = 0 then [
                      LockOut(ctrlPort, rr1);
                      rr1Image := LockIn(ctrlPort);
                      LockOut(ctrlPort, wr3);
                      LockOut(ctrlPort, pCcb^.wr3Init);
                      trb.counter := 0;
                      rxState := rxIdle;
                      pCcb^.rxOn := false;
                      if (rr1Image and crcError) = crcError then
                	 trb.ercRet := ercCrcError
                      else
                         trb.ercRet := ercOK;
                      erc := PSend(complExch, ads (pCcb^.rxIob));
                   ];
                ];

        otherwise
      end;
   ];
end;
{$PAGE+}
{Special receive interrupt service routine is only called upon in instances
 of error.  Close up shop and return a nonzero erc to the requesting process.
 And better luck next time!}

procedure SpRxISR(iLine :sint) [public];
var
   rr0Image, ecsrImage :byte;		{Save data read from 8274}
   ctrlPort :word;			{Reduce pCcb^.ctrl dereferences}
   pCcb :pCcbType;

begin
   pCcb := adr ccb[iLine];		{Establish comm control block}
   with pCcb^ do [
      ctrlPort := ctrl;
      rr0Image := LockIn(ctrlPort);
      LockOut(ctrlPort, errorReset or wr3);
      LockOut(ctrlPort, wr3Init);
      ecsrImage := Input(ecsr);
      rs232C := rs232C * [dataTerminalReady, requestToSend];
      rs232C := rs232C + SetRs232C(rr0Image, ecsrImage, iLine);
      if (rr0Image and rxOverrun = rxOverrun) then
         abortRx(pCcb^, ercRxOverrun)
      else if (rr0Image and parityError = parityError) then
         abortRx(pCcb^, ercParityError);
   ];
end;

end.
