$TITLE	(COMPRESS/DECOMPRESS SUBROUTINES)
;
;	Author: David C. Adams
;
;	April, 1986
;
;	Note:  When assembling, the OBJ file should be "CompressDecompress.obj"
;		and it should be placed in library "HDLCCompress.lib".
;
;	COMPRESS/DECOMPRESS Subroutines
;
;	This module consists of two subroutines as follows:
;
;		COMPRESS: compresses an input string, removing any
;			embedded null characters and replacing the
;			nulls with an encoded value.
;
;		DECOMPRESS: decompresses a string which has been
;			previously encoded by the COMPRESS routine.
;
;	The compression algorithm is designed to compress strings of
;	consecutive NULL and SPACE characters, using a construct referred to as
;	an SCB (String Control Byte).  The format of the SCB depends upon
;	the value of bit 0 (the HIGH ORDER BIT).  If the high order bit is
;	ONE, then the SCB is defined as:
;
;	-------------------------------------------------
;	|  1  |  x  |  x  |  x  |  x  |  x  |  x  |  x  |
;	-------------------------------------------------
;
;	and the bits represented by "x" form a binary count, which can
;	range in value from 1-127.  Note that a value of zero is invalid,
;	and is, in fact, a special case of the SCB:
;
;	-------------------------------------------------
;	|  1  |  0  |  0  |  0  |  0  |  0  |  0  |  0  |
;	-------------------------------------------------
;
;	which is used to indicate that the ENTIRE string is NOT COMPRESSED
;	at all: the string may contain blanks and/or nulls.  The data following
;	this type of SCB has a length equal to the string lenght minus one
;	(to adjust for the X'80' SCB in front).
;
;	If the high order bit of the SCB is ZERO, then the SCB represents
;	a string of compressed spaces or nulls, and this type of SCB is
;	further defined below:
;
;	-------------------------------------------------
;	|  0  |  1  |  x  |  x  |  x  |  x  |  x  |  x  |
;	-------------------------------------------------
;
;	The above SCB represents a string of consecutive SPACES (X'20')
;	from 1-63 bytes in length, as represented by the six low-order
;	bits of the SCB.  Notice that the high order bit is 0 and the
;	next bit is 1.  The other type of compression SCB is:
;
;	-------------------------------------------------
;	|  0  |  0  |  x  |  x  |  x  |  x  |  x  |  x  |
;	-------------------------------------------------
;
;	and in this case (above), the first TWO bits are zero, and the
;	following six bits contain a binary count (1-63) of the number
;	of compressed NULL characters which have been replaced by the SCB.
;
;
;	Note that a string of non-NULL characters is preceded by an SCB
;	that specifies the number of non-NULL characters following it; thus
;	non-NULL strings actually are not compressed: they are expanded
;	by one or more bytes, depending upon the length of the string.
;
;	Null strings and blank strings, however, are replaced by the SCB ;	itself, thus allowing a single byte to replace up to 63 consecutive ;	spaces or nulls. 
;
;	This algorithm was designed to be used to compress data which has
;	a high percentage of NULL (X'00') or SPACE characters (X'20').
;
;	Note that the Decompress algorithm expects that the first byte of
;	its input string is an SCB.  If this is not the case, unpredictable
;	results are bound to happen.
;
;	Both subroutines return an ERC.  In the case of the DECOMPRESS routine,
;	the ERC is always 0.  In the case of the COMPRESS routine, an ERC of 0
;	indicates successful compression, while a non-zero ERC indicates that
;	the output data would have exceeded the maximum size permitted by the
;	caller.  This is a calling program error.
;
$EJECT
	PUBLIC	Compress,Decompress
;
;	STACK usage:  These subroutines utilize 28-30 bytes of stack
;			as depicted below:
;	-------------------------------------------------
;	|     cDESTmax  (Compress call only)		|  +22
;	-------------------------------------------------
;	|     pbSOURCE  (base)				|  +20
;	-------------------------------------------------
;	|     pbSOURCE  (offset)			|  +18
;	-------------------------------------------------
;	|     cbSOURCE					|  +16
;	-------------------------------------------------
;	|     pbDEST	(base)				|  +14
;	-------------------------------------------------
;	|     pbDEST	(offset)			|  +12
;	-------------------------------------------------
;	|     pcbDEST	(base)				|  +10
;	-------------------------------------------------
;	|     pcbDEST	(offset)			|  +8
;	-------------------------------------------------
;	|     Return Address				|  +6
;	-------------------------------------------------
;	|     Return Address				|  +4
;	-------------------------------------------------
;	|     Caller's DS				|  +2
;	-------------------------------------------------
;	|     Caller's BP				|  +0
;	-------------------------------------------------
; --> BP points here after initialization
;	-------------------------------------------------
;	|     OUTCNT					|  -2
;	-------------------------------------------------
;	|     SAVOUT					|  -4
;	-------------------------------------------------
;	|     NULLcnt					|  -6
;	-------------------------------------------------
;	|     SPACEcnt					|  -8
;	-------------------------------------------------
; --> SP points here after initialization

cDESTmax EQU	WORD PTR [BP+22]	; Maximum buffer length (COMP only)
pbSOURCE EQU	WORD PTR [BP+20]	; Pointer to source string
cbSOURCE EQU	WORD PTR [BP+16]	; Length of source string
pbDEST	EQU	WORD PTR [BP+14]	; Pointer to destination string
pcbDEST	EQU	WORD PTR [BP+10]	; Pointer to address of caller's
					; output string length (returned)
;
;	Local variables follow:
;
LOCALln	 EQU	8	; length of local variables for STACK adjustment

OUTCNT	EQU	WORD PTR [BP-2]		; Output string length (calculated)
SAVOUT	EQU	WORD PTR [BP-4]		; Save area for output SCB offset
NULLcnt	EQU	WORD PTR [BP-6]		; NULL character counter
SPACEcnt EQU	WORD PTR [BP-8]		; SPACE character counter
;
;	Equates for housekeeping
;
COMPbytes EQU	16	; number of bytes for RET from Compress routine
DECbytes EQU	14	; number of bytes for RET from Decompress routine
;
;	Handy constants expressed as equates
;
ONE	EQU	1	; CONSTANT OF ONE
ZERO	EQU	0	; CONSTANT OF ZERO
MAXNval	EQU	63	; Maximum value for null SCB length (bits 2-7).
MAXSval	EQU	63	; Maximum value for space SCB length (bits 2-7).
MAXCval	EQU	126	; Maximum value: non-null/space SCB length (bits 1-7)
NULLbyte  EQU	00h	; NULL CHARACTER
SPACEbyte EQU	020h	; SPACE CHARACTER


STACK	SEGMENT STACK 'STACK'
	DW	26 DUP (?)
TSTACK	EQU	$
STACK	ENDS

DGROUP	GROUP	STACK

CompDec	SEGMENT	WORD

ASSUME	CS:CompDec,SS:DGROUP,DS:NOTHING,ES:NOTHING

$EJECT
;
;	Compress subroutine
;
;	Register usage:
;	  AL	- contains input character
;	  BX	- input string length (decremented for overall loop)
;	  CX	- scratch register (used to load ES)
;	  DX	- non-null character count (0-MAXval) [DL]
;	  SI	- input string index (used with ES)
;	  DI	- output string index (used with DS)
;	  ES	- base for input and output strings
;
;	DS:SI - pointer to input (SOURCE) string
;	ES:DI - pointer to output (DEST) string
;
;	Calling sequence:
;		CALL Compress (cDESTmax, pbSOURCE, cbSOURCE, pbDEST, pcbDEST)
;	  where
;		cDESTmax - maximum permitted value of DESTination length
;		pbSOURCE - pointer to first byte of SOURCE string
;		cbSOURCE - length of SOURCE string (word)
;		pbDEST	 - pointer to first location of DESTination string
;		pcbDEST	 - pointer to word which will contain the length
;			   of the DESTination string after compression
;
;	  ERC returned:		0 - Compression completed normally
;
;
;	I N I T I A L I Z A T I O N     --   Load all registers
;
Compress PROC	Far

	PUSH	DS		; save caller's DS register
	PUSH	BP		; and caller's BP register
	MOV	BP,SP		; use current SP value for addresBXng via BP
	SUB	SP,LOCALln	; adjust SP for length of local variables

	MOV	BX,cbSOURCE	; load BX with length of input buffer
	MOV	DX,ZERO		; clear non-null string length counter
	MOV	AX,ZERO		; initialize null counter
	MOV	OUTCNT,ZERO	; initialize output string length counter
	MOV	SPACEcnt,ZERO	; clear space counter
	MOV	NULLcnt,ZERO	; and null counter
	MOV	SI,pbSOURCE-2 	; load SI with offset to SOURCE
	MOV	CX,pbSOURCE	; load CX and then
	MOV	DI,pbDEST-2	; load DI with offset to start of DEST string
	MOV	DS,CX		; DS with base for SOURCE string
	MOV	CX,pbDEST	; load CX and then
	MOV	ES,CX		; ES with base for DEST string
	JMP	CompLOOP	; enter main loop
$EJECT
;
;	Subroutine to check if output exceeds maximum permitted length.
;	This must be coded here so assembler can handle Far reference.
;
CHECKout LABEL  Far
	MOV	CX,cDESTmax	; check if we have
	CMP	OUTCNT,CX	; reached maximum output string length
	JE	Comp13		; yes, then we do special things
	RET			; else return
;
;	Results of compression would exceed maximum length permitted, so
;	instead we create an SCB of X'80' and move the original input
;	buffer "uncompressed" to the output buffer.
;
Comp13:
	POP	AX		; kill local (FAR) return address
	POP	AX		; by POPping stack twice

	MOV	SI,pbSOURCE-2 	; load SI with original offset to SOURCE
	MOV	DI,pbDEST-2	; load DI with original offset to start of DEST
	MOV	CX,080h		; load X'80' into CX
	MOV	ES:[DI],CX	; first byte of destination string is X'80'
	INC	DI		; increment to next position
	MOV	CX,cbSOURCE	; load CX with length of input buffer
	REP MOVSB ; ES:[DI],DS:[SI]  ; move source string as-is
	MOV	CX,cbSOURCE	; load CX with length of input buffer again
	INC	CX		; adjust for the X'80' SCB in front
	MOV	OUTCNT,CX	; put length in OUTCNT
	JMP	Comp12		; and exit to caller
;
;
;
;	**************************************************************
;		Main compression loop
;	**************************************************************
;
CompLOOP:
	MOV	AL,DS:[SI]	; get next byte of user input data in AL
	CMP	AL,NULLbyte	; is byte null ?
	JE	Comp1		; yes
	JMP	Comp4		; no - go handle non-null
;
;	**************************************************************
;		Buffer character was a null ...
;	**************************************************************
;
Comp1:
	MOV	CX,SPACEcnt	; check for
	CMP	CX,ZERO		; accumulated SPACE count
	JE	Comp1A		; zero, continue

	OR	CL,040h		; set high order bits to 01......
	MOV	ES:[DI],CL	; non-zero, store accumulated SPACE count
	INC	DI		; increment output pointer
	INC	OUTCNT		; and byte count
	CALL	CHECKOUT	; check for error condition
	MOV	SPACEcnt,ZERO	; and clear it
Comp1A:
	MOV	CX,NULLcnt	; get null count
	CMP	CX,MAXNval	; is null count at maximum ?
	JNE	Comp2		; no, continue
;
;	AND null count has reached MAXval ...
;
	MOV	ES:[DI],CL	; store null count in output string
	INC	DI		; bump output pointer
	INC	OUTCNT		; and byte count
	CALL	CHECKOUT	; check for error condition
	MOV	NULLcnt,ZERO	; clear null count
;
;	AND null count has not reached MAXval yet ...
Comp2:
	INC	NULLcnt		; add to null count
;
;	common exit from null/space routines
;
Comp2A	LABEL	FAR
	CMP	DL,ZERO		; is non-null count zero ?
	JNE	Comp3		; no, handle it
	JMP	CompEND		; yes -- continue loop
;
;	non-null count was non-zero; store SCB for previous string.
;
Comp3:
	OR	DL,080h		; turn on high order bit of non-null counter
	PUSH	DI		; save DI contents on stack
	MOV	DI,SAVOUT	; get SCB location for non-null string in DI
	MOV	ES:[DI],DL	; store non-null count in saved location
	POP	DI		; restore DI pointer
	MOV	DL,ZERO		; clear non-null counter
	JMP	CompEND		; and continue with next input byte
;
;	Check for SPACE character
;                  
Comp4:
	CMP	AL,SPACEbyte	; is character a space
	JE	Comp4A		; yes
	JMP	Comp5		; no, handle non-null, non-space
Comp4A:
	MOV	CX,NULLcnt	; check for
	CMP	CX,ZERO		; accumulated NULL count
	JE	Comp4B		; zero, continue

	MOV	ES:[DI],CL	; non-zero, store accumulated NULL count
	INC	DI		; bump output count
	INC	OUTCNT		; and byte count
	CALL	CHECKOUT	; check for error condition
	MOV	NULLcnt,ZERO	; and clear it
Comp4B:
	MOV	CX,SPACEcnt	; get space count
	CMP	CX,MAXSval	; is space count at maximum ?
	JNE	Comp4C		; no, continue

	OR	CL,040h		; set high order bits to 01......
	MOV	ES:[DI],CL	; store space count in output string
	INC	DI		; bump output pointer
	INC	OUTCNT		; and byte count
	CALL	CHECKOUT	; check for error condition
	MOV	SPACEcnt,ZERO	; clear space count
Comp4C:
	INC	SPACEcnt	; increment space counter
	JMP	Comp2A		; go through common space/null exit
;
;	**************************************************************
;		Buffer character was a non-null ...
;	**************************************************************
;
Comp5:
	MOV	CX,NULLcnt	; get space count
	CMP	CX,ZERO		; check if null count is zero
	JE	Comp5A		; yes, continue
;
;	AND null count was non-zero, store SCB for null string first ...
;
	MOV	ES:[DI],CL	; put null count (compression byte) in string
	INC	DI		; increment output pointer
	INC	OUTCNT		; and output byte count
	CALL	CHECKOUT	; check for error condition
	MOV	NULLcnt,ZERO	; clear null count
Comp5A:
	MOV	CX,SPACEcnt	; get space count
	CMP	CX,ZERO		; check if SPACE count is zero
	JE	Comp6		; yes, continue
;
;	AND space count was non-zero, store SCB for space string first ...
;
	OR	CL,040h		; set high order bits to 01......
	MOV	ES:[DI],CL	; put space count (compression byte) in string
	INC	DI		; increment output pointer
	INC	OUTCNT		; and output byte count
	CALL	CHECKOUT	; check for error condition
	MOV	SPACEcnt,ZERO	; clear space count
;
;	Check if this is the first non-null character encountered
;
Comp6:
	CMP	DL,ZERO		; is non-null count zero ?
	JNE	Comp7		; no
;
;	First non-null character encountered.  Save SCB index for later use.
;
	MOV	SAVOUT,DI	; save address where flag byte goes
	INC	DI		; bump past flag byte position
	INC	OUTCNT		; and bump output string count
	CALL	CHECKOUT	; check for error condition
;
;	Check if maximum (127) string length reached
;
Comp7:
 	CMP	DL,MAXCval	; have we hit maximum ?
	JNE	Comp8		; no, continue
;
;	Maximum reached - store SCB for previous string
;
	OR	DL,080h		; turn on high order bit
	PUSH	DI		; save output index
	MOV	DI,SAVOUT	; get saved address
	MOV	ES:[DI],DL	; put SCB in output string
	POP	DI		; restore DI pointer
	MOV	DL,ZERO		; clear non-null counter
	JMP	Comp6		; re-enter routine (we haven't touched AL)
;
;	Less than maximum, store character and increment counters
;
Comp8:
	MOV	ES:[DI],AL	; move input byte to area
	INC	DI		; bump output index
	INC	OUTCNT		; increment output string length
	CALL	CHECKOUT	; check for error condition
	ADD	DL,ONE		; increment non-null count
;
;	Finished with this character; Increment input index and decrement count
;
CompEND:
	INC	SI		; increment input index
	DEC	BX		; decrement input string length
	CMP	BX,ZERO		; end of input string ?
	JE	Comp9		; yes, clean up
	JMP	CompLOOP	; no -- continue loop
;
;	***************************************************************
;		End of input data 
;	***************************************************************
;
;	Check for residual null string
;
Comp9:
	CMP	NULLcnt,ZERO	; is null count zero ?
	JE	Comp10		; Yes
;
;	Null count is non-zero, store value in output string
;
	MOV	CX,NULLcnt	; get null count in AX
	MOV	ES:[DI],CL	; stuff null count in output message
	INC	DI		; bump output index
	INC	OUTCNT		; and output byte count
	CALL	CHECKOUT	; check for error condition
Comp10:
	CMP	SPACEcnt,ZERO	; is space count zero ?
	JE	Comp11		; Yes
;
;	Space count is non-zero, store value in output string
;
	MOV	CX,SPACEcnt	; get null count in AX
	OR	CL,040h		; set high order bits to 01......
	MOV	ES:[DI],CL	; stuff null count in output message
	INC	DI		; bump output index
	INC	OUTCNT		; and output byte count
	CALL	CHECKOUT	; check for error condition
;
;	Check for residual non-null string
;
Comp11:	
	CMP	DL,ZERO		; is non-null count zero ?
	JE	Comp12		; yes, need to store it
;
;	Residual non-null string exists, finish handling it.
;
	MOV	DI,SAVOUT	; get non-null index pointer in DI
	OR	DL,080h		; turn on high order bit of count
	MOV	ES:[DI],DL	; store in saved buffer position
;
;	final exit to caller.....
Comp12:
	MOV	BX,pcbDEST-2	; get offset to caller's length word in BX
	MOV	CX,pcbDEST	; and get the segment base
	MOV	ES,CX		; into ES
	MOV	AX,OUTCNT	; length to AX
	MOV	ES:[BX],AX	; then to caller's word
	MOV	AX,ZERO		; set erc=0 and
	ADD	SP,LOCALln	; re-adjust SP
	POP	BP		; get caller's BP value back
	POP	DS		; then restore caller's DS register
	RET	COMPbytes	; exit to caller <--------------- EXIT
	
Compress ENDP

$EJECT
;        
;	Deompress subroutine
;
;	Register usage:
;	  AL	- contains input character
;	  BX	- contains input string length (decremented)
;	  CX	- temporary loop counter
;	  DX	- temporary scratch register (for loading ES)
;	  SI	- input string index
;	  DI	- output string index
;	  ES	- string pointer base
;
;	DS:SI - pointer to input (SOURCE) string
;	ES:DI - pointer to output (DEST) string
;
;	Calling sequence:
;		CALL Decompress (pbSOURCE, cbSOURCE, pbDEST, pcbDEST)
;	  where
;		pbSOURCE - pointer to first byte of SOURCE string.  Note
;			   that this string must be in compressed format.
;		cbSOURCE - length of SOURCE string (word)
;		pbDEST	 - pointer to first location of DESTination string
;		pcbDEST	 - pointer to word which will contain the length
;			   of the DESTination string after decompression.
;
;	ERC returned:  0 = decompresBXon completed
;
Decompress PROC	Far
;
;	initialization of registers
;
	PUSH	DS		; save caller's DS register on stack
	PUSH	BP		; and save caller's BP register
	MOV	BP,SP		; use current SP value for addressing via BP
	SUB	SP,LOCALln	; adjust SP for length of local variables

	MOV	DX,pbSOURCE	; load source string segment base
	MOV	DS,DX		; into DS
	MOV	SI,pbSOURCE-2	; set SI to offset for SOURCE string
	MOV	DX,pbDEST	; load destination string segment base
	MOV	ES,DX		; into ES
	MOV	DI,pbDEST-2	; set DI to offset for DEST string
	MOV 	BX,cbSOURCE	; initialize loop register (# input bytes)
	MOV	AX,ZERO		; clear AX
	MOV	CX,ZERO		; and CX
	MOV 	OUTCNT,ZERO	; clear output string length
;
; 	Check for special case of SCB = X'80'
;
	MOV	AL,DS:[SI]	; get very first SCB
	CMP	AL,080h		; is it X'80' ?
	JNE	GetScb		; no, proceed normally
;
;	**************************************************************
;	Special case of "uncompressed" string
;	**************************************************************
;
	MOV 	CX,cbSOURCE	; initialize loop register (# input bytes)
	DEC	CX		; adjust past SCB
	MOV	OUTCNT,CX	; move to output length counter
	INC	SI		; point past SCB in source string
	REP MOVSB ; ES:[DI],DS:[SI]  ; move input string to output string
	JMP	DECend		; exit to caller
;
;	**************************************************************
;		Main Decompression Loop
;	**************************************************************
GetSCB:	
	CMP	BX,ZERO		; are we done yet ?
	JNZ	Dec1		; no, continue
	JMP	DecEND		; yes, exit to caller
;
;	**************************************************************
;	get an SCB from source string
;	**************************************************************
;
Dec1:
	MOV	AL,DS:[SI]	; get first input byte in AL
	INC	SI		; increment input pointer
	MOV	CL,AL		; get count in CX
	DEC	BX		; decrement input counter
	AND	CL,07Fh		; clear high order bit
	AND	AL,080h		; check high order bit of AL
	JZ	Dec3		; if off, have blanks or nulls
;
;	**************************************************************
;	Have a non-null string following the SCB
;	**************************************************************
;
Dec2:
	ADD	OUTCNT,CX	; add length to OUTCNT
	SUB	BX,CX		; subtract length from total
	REP MOVSB ; ES:[DI],DS:[SI] ; move data, bumping DI and SI
	JMP	GetSCB		; done -- get next SCB
;
;	**************************************************************
;	Have null or space compression SCB
;	**************************************************************
;
Dec3:
	MOV	AL,CL		; get remainder of SCB in AL
	AND	AL,040h		; test second high order bit
	JZ	Dec4		; if it is zero, we have NULL compression
;
;	Expand compressed SPACEs
;
	AND	CL,03Fh		; turn off "space bit" in SCB
	MOV	AL,SPACEbyte	; put SPACE in AL
	JMP	Dec5		; and use same algorithm as for NULLs
;
;	Expand compressed NULLs
;
Dec4:
	MOV	AL,NULLbyte	; put NULL in AL
Dec5:
	ADD	OUTCNT,CX	; add length to OUTCNT
	REP STOSB ; ES:[DI],AL ; fill output string with nulls, incrementing DI
	JMP	GetSCB		; done -- get next SCB
;
;	**************************************************************
;	Loop finished - EXIT to caller, input string exhausted.
;	**************************************************************
;
DecEND:
	MOV	SI,pcbDEST-2	; get offset to callers length field
	MOV	DX,pcbDEST	; and get segment base
	MOV	ES,DX		; into ES
	MOV	AX,OUTCNT	; word with the
	MOV	ES:[SI],AX	; length of the compressed string
	MOV	AX,ZERO		; set erc=0 and
	ADD	SP,LOCALln	; re-adjust SP
	POP	BP		; get caller's BP value back
	POP	DS		; and restore caller's DS register
	RET	DECbytes	; exit to caller <--------------- EXIT


Decompress ENDP	

CompDec	ENDS
	END
