#!/usr/bin/python
# ::::: __________________________________________________________________ :::::
# : ____\ ._ ____ _____ __. ____ ___ _______ .__ ______ .__ _____ .__ _. /____ :
# __\ .___! _\__/__    / _|__   / _/_____  __|  \ gRK __|_ \  __  |_ \ !___. /__
# \   ! ___/  |/  /___/  |   \__\ ._/  __\/  \   \___/  |/  \/  \_./  \___ !   /
# /__  (___   /\____\____|\   ____|   /  /___|\   ______.    ____\|\   ___)  __\
#   /____  \_/ ___________ \_/ __ |__/ _______ \_/ ____ |___/ _____ \_/  ____\
# :     /________________________________________________________________\     :
# :::::       +  p  H  E  N  O  M  p  R  O  D  U  C  T  I  O  N  S  +      :::::
# ==============================================================================
#
# -----------------------------------------
# - modName: mrc_client multiplexer       -
# - majorVersion: 1.2                     -
# - minorVersion: 5                       -
# - author: Stackfault                    -
# - publisher: Phenom Productions         -
# - website: https://www.phenomprod.com   -
# - email: stackfault@bottomlessabyss.net -
# - bbs: bbs.bottomlessabyss.net:2023     -
# -----------------------------------------
#
# Based on previous work from Gryphon of Cyberia BBS
#
# The code have been completely reviewed and improved everywhere I could see
# room for improvement without breaking compatibility.
#
# Major changes:
#
# - Error trapping on all critical locations
# - Socket routine rewrite and now non-blocking
# - Internal auto-restart, no need for an external restart script
# - New commands added and supported by the new server code
# - Message serialization allowing very fast message rate and proper display order
# - Bonus server stats data allowing an in-bbs applet to show MRC status (See samples)
# - Graceful client shutdown notification to the server
# - New BBS information subsystem allowing BBS to provide connection info details
# - New startup check to allow for smoother installation and configuration
#
# Make sure to use the new mrc_config.py so you can take advantage of some new
# features.
#

import os, os.path, sys, fnmatch, glob, re
import time, socket, errno, string, platform

# Import site config
from mrc_config import *

msleep = lambda x: time.sleep(x/1000.0)
bbsdir = os.getcwd()

# Change this info
tempdir  = "%s%stemp"      % (bbsdir,  os.sep)
datadir  = "%s%sdata"      % (bbsdir,  os.sep)
chatdats = "%s%schat*.dat" % (datadir, os.sep)

# Align this path with the MRC MPL (Default: {mrcdatadir}/mrc)
mrcdir   = "%s%smrc"       % (datadir, os.sep)

# Platform information
version        = "1.2.5"
platform_name  = "MYSTIC"
system_name    = platform.system()
machine_arch   = platform.machine()
debugflag      = False
version_string = "%s/%s.%s/%s" % (platform_name, system_name, machine_arch, version)
client_version = "Multi Relay Chat Client v%s [sf]" % version

# Check for command-line args
if(len(sys.argv) < 3) :
    print "Usage : mrc_client.py hostname port"
    sys.exit(1)

# Global vars
host  = sys.argv[1]
port  = int(sys.argv[2])
intv  = [1, 2, 5, 10, 30, 60, 120, 180, 240, 300]   # Auto-restart intervals

# Strip MCI color codes
def stripmci(text):
    return re.sub('\|[0-9]{2}', '', text)

# User chatlog for DLCHATLOG
def chatlog(data):
    if "CLIENT~" not in data and "SERVER~" not in data:
        ltime=time.asctime(time.localtime(time.time()))
        packet   = data.split("~")
        message  = stripmci(packet[6])
        clogfile = "%s%smrcchat.log" % (mrcdir, os.sep)
        clog     = open(clogfile, "a")
        clog.write("%s %s\n" % (ltime, message))
        clog.close()

# Console logger
def logger(loginfo):
  ltime = time.asctime(time.localtime(time.time()))
  print "%s  %s" % (ltime, loginfo.strip())
  sys.stdout.flush()

# Socket sender to server
def send_server(data):
    if data:
        try:
            mrcserver.send(data)
        except:
            logger("Connection error")
            mrcserver.shutdown(2)
            mrcserver.close()

# Temp files cleaning routine
def clean_files():
    mrcfiles = os.listdir( mrcdir )
    for file in mrcfiles:
        if fnmatch.fnmatch(file,'*.mrc'):
            mrcfile = "%s%s%s" % (mrcdir,os.sep,file)
            os.remove(mrcfile)

# Read queued file from MRC, ignoring stale files older than 10s
def send_mrc():
    mrcfiles = os.listdir( mrcdir )
    for file in mrcfiles:
        if fnmatch.fnmatch(file,'*.mrc'):
            mrcfile = "%s%s%s" % (mrcdir,os.sep,file)
            ft = os.path.getmtime(mrcfile)

            # Do not forward packets older than 10s
            if time.time() - ft < 10:
                try:
                    # Avoid reading a file still open by the MPL by
                    # opening it read-write to check for locking
                    f  = open(mrcfile,"r+")
                    fl = f.readline()
                    f.close()
                    mline    = fl.split("~")
                    fromuser = mline[0]
                    message  = mline[6]
                    if message == "VERSION":
                        deliver_mrc("CLIENT~~~%s~~~|07- %s~" % (fromuser, client_version))
                    send_server(fl)
                    if debugflag: logger("< %s" % fl)
                    os.remove(mrcfile)
                except IOError:
                    logger("MRC file still busy")
                    pass
                except:
                    logger("Error:" + fl)
            else:
                os.remove(mrcfile)

# Write time serialized file for display in MRC
def deliver_mrc( server_data ):

    # Make up a serialized filename based on time
    filename = "%s.mrc" % str(int(round(time.time() * 1000)))[5:13]

    try:
        packet   = server_data.split("~")
        fromuser = packet[0]
        fromsite = packet[1]
        fromroom = packet[2]
        touser   = packet[3]
        tosite   = packet[4]
        toroom   = packet[5]
        message  = packet[6]
    except:
        logger("Bad packet: %s" % server_data)
        return

    if debugflag: logger("> %s" % server_data)

    # Manage server PINGs
    if fromuser == "SERVER" and message.lower() == "ping":
        send_im_alive()

    # Manage server stats
    elif fromuser == "SERVER" and message.startswith("STATS:"):
        statsfile = "%s%smrcstats.dat" % (mrcdir, os.sep)
        try:
            f = open(statsfile, "w")
            f.write(message.split(":")[1])
            f.close()
        except:
            logger("Cannot write server stats to %s" % statsfile)

    else:
        chatlog(server_data)
        for f in glob.iglob(chatdats):
            if not 'chatroom' in f:
                chatfile="%s%schat" % (datadir,os.sep)
                xy = f.replace(chatfile,tempdir)
                xy = xy[:-4]
                inusefile = "%s%stchat.inuse" % (xy,os.sep)
                if os.path.isfile(inusefile):
                    mrcfile  = "%s%s%s" % (xy,os.sep,filename)
                    openfile = open(mrcfile,"a")
                    openfile.write(server_data)
                    openfile.close()
                    msleep(20)

# Respond to server PING
def send_im_alive():
    data = "CLIENT~%s~~SERVER~ALL~~IMALIVE:%s~\n" % (bbsname,bbsname)
    send_server(data)

# Send graceful shutdown request to server when exited
def send_shutdown():
    data = "CLIENT~%s~~SERVER~ALL~~SHUTDOWN~\n" % bbsname
    send_server(data)

# Request server stats for applet
def request_stats():
    data = "CLIENT~%s~~SERVER~ALL~~STATS~\n" % bbsname
    send_server(data)

# Send BBS additional info for INFO command
def send_bbsinfo():
    prefix  = "CLIENT~%s~~SERVER~ALL~~" % bbsname
    packet  = prefix + "INFOWEB:%s~\n"  % info_web
    packet += prefix + "INFOTEL:%s~\n"   % info_telnet
    packet += prefix + "INFOSSH:%s~\n"   % info_ssh
    packet += prefix + "INFOSYS:%s~\n"   % info_sysop
    packet += prefix + "INFODSC:%s~\n"   % info_desc
    send_server(packet)

# Handle different line separator scenarios
def check_separator(data):
    if   data.count("\r\n"): return("\r\n")
    elif data.count("\n\r"): return("\n\r")
    elif data.count("\r"):   return("\r")
    else:                    return("\n")

# Main process loop
def mainproc():
    global delay
    global mrcserver

    mrcserver = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    mrcserver.settimeout(5)

    restart    = 0
    readbuffer = ''
    tdat       = ''

    # Non-blocking socket loop to improve speed
    try:
        mrcserver.connect((host, port))
        mrcserver.setblocking(0)
        mrcserver.send("%s~%s" % (bbsname, version_string))
        logger("Connected to Multi Relay Chat host %s port %d" % (host, port))
        delay = 0
    except:
        logger("Unable to connect to %s:%d" % (host,port))
        return

    send_bbsinfo()
    send_im_alive()

    loop = 1400
    while True:
        msleep(20)
        send_mrc()

        loop += 1
        try:
            readbuffer = mrcserver.recv(4096)
        except socket.error, e:
            err = e.args[0]
            if err == errno.EAGAIN or err == errno.EWOULDBLOCK:
                # Request stats every 40 seconds
                if loop % 2000 == 0:
                    request_stats()
                    loop = 0
                continue
            else:
                restart = 1
        else:
            if readbuffer:
                sep  = check_separator(readbuffer)
                tdat = readbuffer.split(sep)
                for data in tdat:
                    if data:
                        deliver_mrc(data)
            else:
                restart = 1

        # Handle socket restarts with socket shutdowns
        if restart:
            logger("Lost connection to server\n")
            mrcserver.shutdown(2)
            mrcserver.close()
            return

# Some validation of config to ensure smoother operation
def check_startup():
    failed = 0

    if not os.path.exists(mrcdir):
        os.makedirs(mrcdir)

    if len(stripmci(bbsname)) < 5:
        print "Config: 'bbsname' should be set to something sensible"
        failed = 1

    if len(stripmci(bbsname)) > 40:
        print "Config: 'bbsname' cannot be longer than 40 characters after PIPE codes evaluation"
        failed = 1

    for param in ['info_web' 'info_telnet', 'info_ssh', 'info_sysop', 'info_desc']:
        if len(stripmci(param)) > 64:
            print "Config: '%s' cannot be longer than 64 characters after PIPE codes evaluation" % param
            failed = 1

    for param in ['info_web' 'info_telnet', 'info_ssh', 'info_sysop', 'info_desc']:
        if len(param) > 128:
            print "Config: '%s' cannot be longer than 128 characters including PIPE codes" % param
            failed = 1

    if failed:
        print "This must be fixed in mrc_config.py"
        sys.exit()

# Main loop
if __name__ == "__main__":
    logger(client_version)
    check_startup()
    delay = 0
    while True:
        try:
            mainproc()

            # Incremental auto-restart built-in
            logger("Reconnecting in %d seconds" % intv[delay])
            time.sleep(intv[delay])
            delay += 1
            if delay > 9: delay = 0

        except KeyboardInterrupt:
            logger("Shutting down")
            try:
                send_shutdown()
                mrcserver.shutdown(2)
            finally:
                mrcserver.close()
            sys.exit()
        except:
            continue

