#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mpg123.h"

#ifdef READ_MMAP
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED ( (void *) -1 )
#endif
#endif

static int get_fileinfo(struct reader *,char *buf);


/*******************************************************************
 * stream based operation
 */


/*  buffered file read routine  (C) by GyikSoft  */ 
#define bsr_bufsize (512*1024)
long bsr_buffersize=bsr_bufsize;
unsigned char bsr_buffer[bsr_bufsize];
off_t bsr_pt,bsr_startpt,bsr_endpt,bsr_filesize;

void bsr_update(int fd,off_t o){
  lseek(fd,o,SEEK_SET);
  bsr_startpt=bsr_pt=o;
  if( (bsr_endpt=read(fd,bsr_buffer,bsr_buffersize)) >=0 ) bsr_endpt+=o;
  if(bsr_endpt>bsr_filesize) bsr_endpt=bsr_filesize;
}

void bsr_init(int fd){
   fprintf(stderr,"Using GyikSoft's buffered I/O routines for this stream.\015\n");
/*    if((bsr_filesize=lseek(fd,0,SEEK_END))<0)bsr_filesize=0;  */
    bsr_pt=0;
    bsr_startpt=0;
    bsr_endpt=0;
    bsr_update(fd,0);
}

off_t bsr_lseek(int fd,off_t o){
  if(o<0)return(o);
  if((o>=bsr_startpt)&&(o<=bsr_endpt))return(bsr_pt=o);
  bsr_update(fd,o);
  return(bsr_pt);
}

ssize_t bsr_read(int fd,void *buf,size_t count){
  long int x,y;
  if( (x=bsr_endpt-bsr_pt) >0 ){
    if(x>count) x=count;
    memcpy(buf,bsr_buffer+bsr_pt-bsr_startpt,x);
    bsr_pt+=x;
    if((count-=x)==0) return(x);  /* beolvasta az egeszet jol */
  } else x=0;

  bsr_update(fd,bsr_pt);

  if( (y=bsr_endpt-bsr_pt) >0 ){
    if(y>count) y=count;
    memcpy(buf+x,bsr_buffer+bsr_pt-bsr_startpt,y);
    bsr_pt+=y;
  } else y=0;
  return(x+y);
}


         /*  stream routines  */


static int fullread(int fd,unsigned char *buf,int count)
{
  int ret,cnt=0;

  while(cnt < count) {
    ret = bsr_read(fd,buf+cnt,count-cnt);
    if(ret < 0)
      return ret;
    if(ret == 0)
      break;
    cnt += ret;
  } 

  return cnt;
}

static int default_init(struct reader *rds)
{
  int len;
  char buf[128];
  
  len = get_fileinfo(rds,buf);

  bsr_init(rds->filept);
  bsr_filesize=len;
  
  rds->filelen = -1;
  rds->filepos = 0;
  
  if(len > 0) {
    if(!strncmp(buf,"TAG",3)) {
      rds->flags |= READER_ID3TAG;
      memcpy(rds->id3buf,buf,128);
    }
    rds->filelen = len;
    rds->filepos = 0;
  }
  return 0;
}

void stream_close(struct reader *rds)
{
    if (rds->flags & READER_FD_OPENED)
        close(rds->filept);
}

/**************************************** 
 * HACK,HACK,HACK: step back <num> frames 
 * can only work if the 'stream' isn't a real stream but a file
 */
static int stream_back_bytes(struct reader *rds,int bytes)
{
  if(bsr_lseek(rds->filept,bsr_pt-bytes) < 0) return -1;
  return 0;
}

static int stream_head_read(struct reader *rds,unsigned long *newhead)
{
  unsigned char hbuf[4];

  if(fullread(rds->filept,hbuf,4) != 4)
    return FALSE;
  
  *newhead = ((unsigned long) hbuf[0] << 24) |
    ((unsigned long) hbuf[1] << 16) |
    ((unsigned long) hbuf[2] << 8)  |
    (unsigned long) hbuf[3];
  
  return TRUE;
}

static int stream_head_shift(struct reader *rds,unsigned long *head)
{
  unsigned char hbuf;

  if(fullread(rds->filept,&hbuf,1) != 1)
    return 0;
  *head <<= 8;
  *head |= hbuf;
  *head &= 0xffffffff;
  return 1;
}

static int stream_skip_bytes(struct reader *rds,int len)
{
  return bsr_lseek(rds->filept,bsr_pt+len);
}

static int stream_read_frame_body(struct reader *rds,unsigned char *buf,
				  int size)
{
  long l;

/*  if((bsr_pt+size)>bsr_filesize) return FALSE;  */

  if( (l=fullread(rds->filept,buf,size)) != size) return FALSE;

/*
  {
    if(l <= 0)
      return 0;
    memset(buf+l,0,size-l);
  }
*/

  return TRUE;
}

static long stream_tell(struct reader *rds)
{
  return bsr_pt; /* lseek(rds->filept,0,SEEK_CUR); */
}

static void stream_rewind(struct reader *rds)
{
  bsr_lseek(rds->filept,0);
}

/*
 * returns length of a file (if filept points to a file)
 * reads last to 128 bytes information into buffer
 */
static int get_fileinfo(struct reader *rds,char *buf)
{
	int len;

        if((len=lseek(rds->filept,0,SEEK_END)) < 0) {
                return -1;
        }
        if(lseek(rds->filept,-128,SEEK_END) < 0)
                return -1;
        if(read(rds->filept,(unsigned char *)buf,128) != 128) {
                return -1;
        }
        if(!strncmp(buf,"TAG",3)) {
                len -= 128;
        }
        if(lseek(rds->filept,0,SEEK_SET) < 0)
                return -1;
        if(len <= 0)
                return -1;
	return len;
}


#ifdef READ_MMAP
/*********************************************************+
 * memory mapped operation 
 *
 */
static unsigned char *mapbuf;
static unsigned char *mappnt;
static unsigned char *mapend;

static int mapped_init(struct reader *rds) 
{
	long len;
	char buf[128];

	len = get_fileinfo(rds,buf);
	if(len < 0)
		return -1;

	if(!strncmp(buf,"TAG",3)) {
	  rds->flags |= READER_ID3TAG;
	  memcpy(rds->id3buf,buf,128);
	}

        mappnt = mapbuf = 
		mmap(NULL, len, PROT_READ, MAP_SHARED , rds->filept, 0);
	if(!mapbuf || mapbuf == MAP_FAILED)
		return -1;

	mapend = mapbuf + len;
	
/*	if(param.verbose > 1)  */
		fprintf(stderr,"Using memory mapped IO for this stream.\n");

	rds->filelen = len;
	return 0;
}

static void mapped_rewind(struct reader *rds)
{
	mappnt = mapbuf;
}

static void mapped_close(struct reader *rds)
{
	munmap(mapbuf,mapend-mapbuf);
	if (rds->flags & READER_FD_OPENED)
		close(rds->filept);
}

static int mapped_head_read(struct reader *rds,unsigned long *newhead) 
{
	unsigned long nh;

	if(mappnt + 4 > mapend)
		return FALSE;

	nh = (*mappnt++)  << 24;
	nh |= (*mappnt++) << 16;
	nh |= (*mappnt++) << 8;
	nh |= (*mappnt++) ;

	*newhead = nh;
	return TRUE;
}

static int mapped_head_shift(struct reader *rds,unsigned long *head)
{
  if(mappnt + 1 > mapend)
    return FALSE;
  *head <<= 8;
  *head |= *mappnt++;
  *head &= 0xffffffff;
  return TRUE;
}

static int mapped_skip_bytes(struct reader *rds,int len)
{
  if(mappnt + len > mapend)
    return FALSE;
  mappnt += len;
  return TRUE;
}

static int mapped_read_frame_body(struct reader *rds,unsigned char *buf,
				  int size)
{
#if 1
  if(mappnt + size > mapend)
    return FALSE;
#else
  long l;
  if(size > (mapend-mappnt)) {
    l = mapend-mappnt;
    memcpy(buf,mappnt,l);
    memset(buf+l,0,size-l);
  }
  else
#endif
    memcpy(buf,mappnt,size);

  mappnt += size;

  return TRUE;
}

static int mapped_back_bytes(struct reader *rds,int bytes)
{
    if( (mappnt - bytes) < mapbuf || (mappnt - bytes + 4) > mapend)
        return -1;
    mappnt -= bytes;
    return 0;
}

#if 0
static int mapped_back_frame(struct reader *rds,struct frame *fr,int num)
{
    long bytes;
    unsigned long newhead;

    if(!firsthead)
        return 0;

    bytes = (fr->framesize+8)*(num+2);

    if( (mappnt - bytes) < mapbuf || (mappnt - bytes + 4) > mapend)
        return -1;
    mappnt -= bytes;

    newhead = (mappnt[0]<<24) + (mappnt[1]<<16) + (mappnt[2]<<8) + mappnt[3];
    mappnt += 4;

    while( (newhead & HDRCMPMASK) != (firsthead & HDRCMPMASK) ) {
        if(mappnt + 1 > mapend)
            return -1;
        newhead <<= 8;
        newhead |= *mappnt++;
        newhead &= 0xffffffff;
    }
    mappnt -= 4;

    read_frame(fr);
    read_frame(fr);

    if(fr->lay == 3)
        set_pointer(512);

    return 0;
}
#endif

static long mapped_tell(struct reader *rds)
{
  return mappnt - mapbuf;
}

#endif

/*****************************************************************
 * read frame helper
 */

struct reader *rd;
struct reader readers[] = {
#ifdef READ_SYSTEM
 { system_init,
   NULL,	/* filled in by system_init() */
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL } ,
#endif
#ifdef READ_MMAP
 { mapped_init,
   mapped_close,
   mapped_head_read,
   mapped_head_shift,
   mapped_skip_bytes,
   mapped_read_frame_body,
   mapped_back_bytes,
   mapped_tell,
   mapped_rewind } ,
#endif
 { default_init,
   stream_close,
   stream_head_read,
   stream_head_shift,
   stream_skip_bytes,
   stream_read_frame_body,
   stream_back_bytes,
   stream_tell,
   stream_rewind } ,
 { NULL, }
};


/* open the device to read the bit stream from it */

void open_stream(char *bs_filenam,int fd)
{
    int i;
    int filept_opened = 1;
    int filept;

    if (!bs_filenam) {
		if(fd < 0) {
			filept = 0;
			filept_opened = 0;
		}
		else
			filept = fd;
	}
	else if (!strncmp(bs_filenam, "http://", 7)) 
		filept = http_open(bs_filenam);
	else if ( (filept = open(bs_filenam, O_RDONLY)) < 0) {
		perror (bs_filenam);
		exit(1);
	}

    rd = NULL;
    for(i=0;;i++) {
      readers[i].filelen = -1;
      readers[i].filept  = filept;
      readers[i].flags = 0;
      if(filept_opened)
        readers[i].flags |= READER_FD_OPENED;
      if(!readers[i].init) {
	fprintf(stderr,"Fatal error!\n");
	exit(1);
      }
      if(readers[i].init(readers+i) >= 0) {
        rd = &readers[i];
        break;
      }
    }

    if(rd && rd->flags & READER_ID3TAG) {
      print_id3_tag(rd->id3buf);
    }
}
















